<?php

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'Pwf_Woo_Product_Variations' ) ) {

	/**
	 *
	 * @since 1.6.6
	 */

	class Pwf_Woo_Product_Variations {

		/**
		* The unique instance of the Pwf_Parse_Query_Vars.
		*
		* @var Pwf_Parse_Query_Vars
		*/
		protected $parse_query_vars;

		/**
		 * Taxonomy Terms
		 *
		 * @since 1.6.6
		 */
		protected $outofstock_terms;

		/**
		 * Fliter post ID
		 *
		 * @since 1.6.6
		 */
		protected $filter_id;

		protected $out_of_stock_ids;

		public function __construct( array $outofstock_terms, Pwf_Parse_Query_Vars $parse_query_vars ) {
			$this->filter_id        = $parse_query_vars->get_filter_id();
			$this->outofstock_terms = $outofstock_terms;
			$this->parse_query_vars = $parse_query_vars;

			$this->out_of_stock_ids = $this->out_of_stock_ids();
		}

		public function get_out_of_stock_ids() {
			return $this->out_of_stock_ids;
		}

		protected function out_of_stock_ids() {
			$args = array(
				'values'       => array(),
				'attrs_filter' => array(),
				'direct_db'    => true,
				'filter_id'    => $this->filter_id,
			);

			foreach ( $this->outofstock_terms as $term ) {
				$slugs = array();
				foreach ( $term['terms'] as $term_id ) {
					$current_term = get_term_by( 'id', $term_id, $term['taxonomy'], ARRAY_A );
					if ( $current_term ) {
						array_push( $args['values'], $current_term['slug'] );
						array_push( $slugs, $current_term['slug'] );
					}
				}
				if ( ! empty( $slugs ) ) {
					$args['attrs_filter'][ $term['taxonomy'] ] = array(
						'slugs'    => $slugs,
						'operator' => $term['operator'],
					);
				}
			}

			if ( 1 !== count( $args['values'] ) ) {
				foreach ( $args['attrs_filter'] as $key => $attr ) {
					if ( 'IN' === $attr['operator'] ) {
						$args['direct_db'] = false;
						break;
					}
				}
			}

			return $this->get_out_of_stock_product_variations_ids( $args );
		}

		/**
		 * Set database query to get out of stock product IDs
		 * @param array $args have all selected attributes
		 * @since 1.4.8, 1.4.9, 1.6.6
		 *
		 * @return array out of stock product_ids
		 */
		protected function get_out_of_stock_product_variations_ids( $args ) {
			global $wpdb;

			if ( $args['direct_db'] ) {
				$query = 'SELECT DISTINCT t1.post_parent ';
			} else {
				$query = 'SELECT t1.post_parent AS post_parent, t1.post_id AS post_id, t1.meta_key AS meta_key, t1.meta_value AS meta_value, t1.stock_status AS stock_status, t20.count_attr, t30.count_out_of_stock';
			}
			$help_query = $this->sql_help_query( $args );

			$query .= "
			FROM (
				SELECT *
				{$help_query}
			) AS t1
			JOIN (
				SELECT COUNT(*) AS count_attr, t2.post_parent AS t2_post_parent, t3.post_id AS t3_post_id, t3.meta_key AS t3_meta_key, t3.meta_value AS t3_meta_value
				{$help_query}
				GROUP BY t2.post_parent, t3.meta_key, t3.meta_value
			) AS t20 ON t20.t2_post_parent = post_parent AND meta_key = t20.t3_meta_key AND t20.t3_meta_value = meta_value
			LEFT JOIN (
				SELECT COUNT(*) AS count_out_of_stock, t2.post_parent AS t2_post_parent, t3.post_id AS t3_post_id, t3.meta_key AS t3_meta_key, t3.meta_value AS t3_meta_value
				{$this->sql_help_query( $args, true )}
				GROUP BY t2.post_parent, t3.meta_key, t3.meta_value
			) AS t30 ON t30.t2_post_parent = post_parent AND meta_key = t30.t3_meta_key AND t30.t3_meta_value = meta_value
			";

			if ( $args['direct_db'] ) {
				$query .= " WHERE stock_status = 'outofstock' AND count_attr = count_out_of_stock";
			}
			$query .= ' ORDER BY post_parent';

			// We have a query - let's see if cached results of this query already exist.
			$query_hash = md5( $query );

			// Maybe store a transient of the count values.
			$cache = apply_filters( 'pwf_woo_filter_count_maybe_cache', true );
			if ( true === $cache ) {
				$cached_counts = (array) get_transient( 'pwf_woo_filter_item_get_outofstock_variation' );
			} else {
				$cached_counts = array();
			}

			if ( ! isset( $cached_counts[ $query_hash ] ) ) {
				$results                      = $wpdb->get_results( $query, ARRAY_A ); // @codingStandardsIgnoreLine
				if ( ! $args['direct_db'] ) {
					$products      = self::build_products_hierarchical( $results );
					$out_of_stocks = array();
					foreach ( $args['attrs_filter'] as $arg ) {
						$out_of_stocks = array_merge( $out_of_stocks, self::filtering_out_of_stock_ids( $products, $arg ) );
					}
					$counts = array_unique( $out_of_stocks );
				} else {
					$counts = array_map( 'absint', wp_list_pluck( $results, 'post_parent' ) );
				}
				$cached_counts[ $query_hash ] = $counts;
				if ( true === $cache ) {
					$transient_time = get_option( 'pwf_transient_time', 86400 );
					set_transient( 'pwf_woo_filter_item_get_outofstock_variation', $cached_counts, $transient_time );
				}
			}

			return array_map( 'absint', (array) $cached_counts[ $query_hash ] );
		}

		/**
		 * Get default query by the selected options for End user
		 * @since 1.4.9, 1.4.6
		 *
		 * @return string sql
		 */
		protected function sql_help_query( $args, $set_out_of_stock = false ) {
			global $wpdb;

			$meta_query     = new WP_Meta_Query( $this->parse_query_vars->get_meta_query() ); // Use this when you need to filter with post meta
			$tax_query      = Pwf_Render_Filter::get_tax_query_class( $this->parse_query_vars->get_tax_query(), 'product' );
			$meta_query_sql = $meta_query->get_sql( 'post', $wpdb->posts, 'ID' );
			$tax_query_sql  = $tax_query->get_sql( $wpdb->posts, 'ID' );

			$sql_where  = $tax_query_sql['where'];
			$sql_where .= $meta_query_sql['where'];

			if ( ! empty( $this->parse_query_vars->get_date_query() ) ) {
				$sql_where .= Pwf_Db_Utilities::get_date_where_sql( $this->parse_query_vars->get_date_query() );
			}
			if ( ! empty( $this->parse_query_vars->get_search_query() ) ) {
				$sql_where .= Pwf_Db_Utilities::get_search_where_sql( $this->parse_query_vars->get_search_query(), $this->filter_id, $this->parse_query_vars, $this->parse_query_vars->get_search_inside() );
			}
			if ( ! empty( $this->parse_query_vars->get_authors_id() ) ) {
				$sql_where .= Pwf_Db_Utilities::get_authors_where_sql( $this->parse_query_vars->get_authors_id() );
			}
			if ( $this->parse_query_vars->has_on_sale() ) {
				$sql_where .= Pwf_Db_Utilities::get_on_sale_where_sql();
			}

			$sql_join = $tax_query_sql['join'] . $meta_query_sql['join'];

			if ( $this->parse_query_vars->has_price_item() ) {
				$price      = $this->parse_query_vars->get_current_min_max_price();
				$sql_join  .= Pwf_Db_Utilities::get_price_join_sql();
				$sql_where .= Pwf_Db_Utilities::get_price_where_sql( $price[0], $price[1] );
			}

			$sql =
			"FROM ( 
				SELECT DISTINCT {$wpdb->posts}.ID AS post_parent
				FROM {$wpdb->posts}"
				. $sql_join
				. ' ' . Pwf_Db_Utilities::get_post_type_where_sql( 'product', $this->filter_id )
				. $sql_where
				. "
			) AS t2
			JOIN (
				SELECT t11_posts.ID AS post_id, post_parent AS t2_post_parent, t12.meta_key, t12.meta_value, t13.meta_value AS stock_status
				FROM {$wpdb->posts} AS t11_posts
				JOIN {$wpdb->postmeta} AS t12 ON t11_posts.ID = t12.post_id AND t12.meta_key LIKE 'attribute_pa_%' AND t12.meta_value IN ('"
				. implode( "','", array_map( 'esc_attr', $args['values'] ) ) . "')
				JOIN {$wpdb->postmeta} AS t13 ON t11_posts.ID = t13.post_id AND t13.meta_key = '_stock_status'";
			if ( true === $set_out_of_stock ) {
				$sql .= " AND t13.meta_value = 'outofstock'";
			}
			$sql .= '
				GROUP BY post_parent, post_id, meta_key, meta_value	
			) AS t3 ON t2.post_parent = t3.t2_post_parent';

			return $sql;
		}

		/**
		 * Build nested array Each parent product have childern 'childern is prodcut_variations
		 * @since 1.4.8
		 *
		 *  @return array product childern inside parent product
		 */
		protected static function build_products_hierarchical( $products ) {
			$results = array();

			foreach ( $products as $product ) {
				$results[ 'parent_id_' . $product['post_parent'] ][] = $product;
			}

			return $results;
		}

		/**
		 * Check if the operator is IN OR AND for each attribue
		 * If is operator is 'IN' means this product contain out of stock for all attributes so we add ti to out_of_stocks variable
		 * @since 1.4.8
		 *
		 * @return array out of stock product_ids
		 */
		protected static function filtering_out_of_stock_ids( &$products, $args ) {
			$out_of_stock_ids = array();

			foreach ( $products as $key => $children ) {
				if ( 1 === count( $children ) ) {
					if ( 'outofstock' === $children[0]['stock_status'] ) {
						array_push( $out_of_stock_ids, $children[0]['post_parent'] );
						unset( $products[ 'parent_id_' . $children[0]['post_parent'] ] );
					}
				} else {
					$check_is_out = array();
					foreach ( $children as $product ) {
						if ( in_array( $product['meta_value'], $args['slugs'], true ) ) {
							if ( 'outofstock' === $product['stock_status'] && $product['count_attr'] === $product['count_out_of_stock'] ) {
								if ( 'IN' === $args['operator'] ) {
									array_push( $check_is_out, $product['meta_value'] );
								} else {
									array_push( $out_of_stock_ids, $children[0]['post_parent'] );
									unset( $products[ 'parent_id_' . $children[0]['post_parent'] ] );
								}
							}
						}
					}

					if ( ! empty( $check_is_out ) && count( array_unique( $check_is_out ) ) === count( $args['slugs'] ) ) {
						array_push( $out_of_stock_ids, $product['post_parent'] );
						unset( $products[ 'parent_id_' . $product['post_parent'] ] );
					}
				}
			}

			return $out_of_stock_ids;
		}
	}
}
