<?php

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'Pwf_Seo' ) ) {
	/**
	 * @since 1.7.8
	 */
	class Pwf_Seo {

		/**
		 * Current page type maybe archive, taxonomy, and page
		 */
		protected static $current_page_type;

		protected static $current_page_id;

		/**
		 * Contain all apges rules saved in Database
		 */
		protected static $all_pages_rules;

		/**
		 * Hold all rules related to this page
		 * @param array $group_rules
		 */
		protected static $group_rules = array();

		/**
		 * URL string hold selected options by user
		 * @param string selected_options
		 */
		protected static $url_for_selected_options;

		/**
		 * URL clean url hold selected option by user
		 */
		protected static $clean_url_selected_options;

		/**
		 * Hold current rewrite rule data info
		 */
		protected static $rewrite_rule_data;

		/**
		 * Hold current page title
		 */
		protected static $page_title;

		protected static $canoncial_url;

		protected static $separator = '%';

		/**
		 * @param array $args check class hook_wp_query $global_args
		 */
		public function __construct( $args ) {
			if ( ! self::is_plugin_activated() ) {
				return;
			}

			self::$all_pages_rules = self::get_seo_rules();

			if ( empty( self::$all_pages_rules ) || ! is_array( self::$all_pages_rules ) ) {
				return;
			}

			self::$current_page_id   = $args['page_id'];
			self::$current_page_type = $args['page_type'];
			if ( empty( self::$current_page_type ) ) {
				return;
			}

			self::$group_rules = self::set_group_rules();
			if ( empty( self::$group_rules ) ) {
				return;
			}

			/**
			 * Need to add compitable to clean URL
			 */
			self::$url_for_selected_options = self::build_selected_options_string();
			if ( empty( self::$url_for_selected_options ) ) {
				return;
			}

			self::convert_rewrite_rules_to_pattern();
			self::$rewrite_rule_data = apply_filters( 'pwf_seo_rewrite_rule_data', self::get_rewrite_rule_data() );
			if ( ! empty( self::$rewrite_rule_data ) ) {
				$GLOBALS['pwf_main_query']['seo_data_info'] = self::$rewrite_rule_data;
				$this->add_seo_data();
			}
		}

		/**
		 * Check if Pretty Enable && current filter use Pretty
		 *
		 * @return bool Is pretty enable.
		 */
		protected static function is_pretty() {
			$is_pretty     = false;
			$pretty_option = get_option( 'pwf_shop_enable_pretty_links', 'enable' );
			if ( 'enable' === $pretty_option && 'on' === $GLOBALS['pwf_main_query']['filter_settings']['pretty_urls'] ) {
				$is_pretty = true;
			}

			return $is_pretty;
		}

		/**
		 * Set seo rules those are saved in option Database
		 *
		 * @return array|empty Seo rules
		 */
		protected static function get_seo_rules() {
			return get_option( 'pwf_woo_filter_seo_rules_options', '' );
		}

		/**
		 * Set group rules for the current page
		 * May be one page has multi rules
		 *
		 * @return array Current page Seo rules
		 */
		protected static function set_group_rules() {
			$results = array();
			foreach ( self::$all_pages_rules as $page_rules ) {
				if ( ! self::chech_page_rule_language( $page_rules['lang'] ) ) {
					continue;
				}
				$page_type = $page_rules['page_type'];
				$split     = explode( '__', $page_type );
				$add_rules = false; // Check to add these rules or no

				if ( 'archive' === self::$current_page_type && count( $split ) === 2 && 'archive' === $split[1] ) {
					if ( is_post_type_archive( $split[0] ) || is_home() ) {
						$add_rules = true;
					} elseif ( is_home() ) {
						$add_rules = true;
					}
				} elseif ( 'page' === self::$current_page_type && count( $split ) === 2 && 'page' === $split[0] ) {
					$page_id = Pwf_Wpml::get_translated_term_id( absint( $split[1] ), 'page' );
					if ( absint( $page_id ) === get_queried_object_id() ) {
						$add_rules = true;
					}
				} elseif ( 'taxonomy' === self::$current_page_type && count( $split ) === 3 ) {
					// All or term _id
					$post_type = $split[0];
					$tax_name  = $split[1];
					if ( 'all' === $split[2] ) {
						if ( 'post' === $post_type && 'category' === $tax_name && is_category() ) {
							$add_rules = true;
						} elseif ( 'post' === $post_type && 'tag' === $tax_name && is_tag() ) {
							$add_rules = true;
						} elseif ( is_tax( $tax_name ) ) {
							$add_rules = true;
						}
					} else {
						$tax_id = $split[2];
						$tax_id = Pwf_Wpml::get_translated_term_id( absint( $tax_id ), esc_attr( $tax_name ) );
						if ( 'post' === $post_type && 'category' === $tax_name && is_category( $tax_id ) ) {
							$add_rules = true;
						} elseif ( 'post' === $post_type && 'tag' === $tax_name && is_tag( $tax_id ) ) {
							$add_rules = true;
						} elseif ( is_tax( $tax_name, $tax_id ) ) {
							$add_rules = true;
						}
					}
				}

				if ( $add_rules ) {
					foreach ( $page_rules['rules'] as $rule ) {
						array_push( $results, $rule );
					}
				}
			}

			return $results;
		}

		/**
		 * Check is Multi language plugin is defined and current page rule
		 * has that language
		 *
		 * @return boolean
		 */
		protected static function chech_page_rule_language( $page_lang ) {
			if ( empty( $page_lang ) ) {
				return true;
			}
			if ( Pwf_Wpml::is_wpml_defined() ) {
				$current_language = ICL_LANGUAGE_CODE;
				if ( $current_language === $page_lang ) {
					return true;
				} else {
					return false;
				}
			} else {
				return true;
			}
		}

		/**
		 * Build selected options string from selected options
		 *
		 * @return string selected options by client
		 */
		protected static function build_selected_options_string() {
			$result       = array();
			$clean_result = array(); // for example, &product-category=clothing
			/**
			 * @param $user_options using it to get slug.
			 * @param $selected_options using it to get exactly order.
			 */
			$user_options     = $GLOBALS['pwf_main_query']['selected_options'] ?? array();
			$parse_query_vars = $GLOBALS['pwf_main_query']['parse_query_vars'];
			$selected_options = $parse_query_vars->selected_items();
			foreach ( $selected_options as $url_key => $data ) {
				if ( isset( $user_options[ $url_key ] ) && is_array( $user_options[ $url_key ] ) ) {
					if ( 'search' === $data['field_type'] ) {
						$result[]       = $url_key . '-for-' . esc_attr( $user_options[ $url_key ][0] );
						$clean_result[] = $url_key . '=' . esc_attr( $user_options[ $url_key ][0] );
					} elseif ( ( 'priceslider' === $data['field_type'] || 'rangeslider' === $data['field_type'] ) && count( $data['values'] ) === 2 ) {
						if ( ! self::is_pretty() && 'two' === $data['slider_info']['format'] ) {
							$clean_result[] = $data['slider_info']['url_key_min'] . '=' . esc_attr( $data['values'][0] );
							$clean_result[] = $data['slider_info']['url_key_max'] . '=' . esc_attr( $data['values'][1] );
						} else {
							$clean_result[] = $url_key . '=' . esc_attr( $data['values'][0] ) . '-' . esc_attr( $data['values'][1] );
						}
						$result[] = $url_key . '-' . esc_attr( $data['values'][0] ) . '-to-' . esc_attr( $data['values'][1] );
					} elseif ( count( $data['values'] ) === 1 && isset( $user_options[ $url_key ] ) ) {
						$result[]       = $url_key . '-' . esc_attr( $user_options[ $url_key ][0] );
						$clean_result[] = $url_key . '=' . esc_attr( $user_options[ $url_key ][0] );
					}
				}
			}

			if ( ! empty( $result ) ) {
				$result  = implode( '/', $result );
				$result .= '/';

				// used with clean url product-cat=clothing
				self::$clean_url_selected_options = implode( '&', $clean_result );
			}

			return $result;
		}

		/**
		 * Convert saved options rewrite rule entered by client to Regular expressions
		 * Replace %any% with Regular expressions pattern
		 * Replace %string% by remove % and % from default string
		 *
		 * @return none
		 */
		protected static function convert_rewrite_rules_to_pattern() {
			foreach ( self::$group_rules as $i => $rewrite_rule ) {
				$rewrite_rule_arr = explode( '/', $rewrite_rule['rewrite_rule'] );
				if ( empty( $rewrite_rule_arr[ count( $rewrite_rule_arr ) - 1 ] ) ) {
					unset( $rewrite_rule_arr[ count( $rewrite_rule_arr ) - 1 ] );
				}
				foreach ( $rewrite_rule_arr as $index => $part_of_rule ) {
					if ( false !== strrpos( $part_of_rule, self::$separator . 'any' ) ) {
						// any
						$rewrite_rule_arr[ $index ] = str_replace( self::$separator . 'any' . self::$separator, '[A-Za-z0-9\-\_]+', $part_of_rule );
					} else {
						$rewrite_rule_arr[ $index ] = str_replace( array( self::$separator ), '', $part_of_rule );
					}
				}
				self::$group_rules[ $i ]['rewrite_rule'] = '#^' . implode( '/', $rewrite_rule_arr ) . '/$#';
			}
		}

		/**
		 * Check current URL [Selected options by client] against saved seo rewrite rules
		 *
		 * @return array
		 */
		protected static function get_rewrite_rule_data() {
			$rewrite_rule_info = array();
			foreach ( self::$group_rules as $index => $rewrite_rule ) {
				if ( preg_match( $rewrite_rule['rewrite_rule'], self::$url_for_selected_options ) ) {
					$rewrite_rule_info = $rewrite_rule;
					break;
				}
			}

			return $rewrite_rule_info;
		}

		/**
		 * Get current page title
		 * @see wp_get_document_title()
		 */
		public static function get_current_page_title() {
			if ( is_front_page() && is_home() ) {
				$title = get_bloginfo( 'name', 'display' );
			} elseif ( is_front_page() ) {
				$page_id = get_option( 'page_on_front' );
				$title   = get_the_title( absint( $page_id ) );
			} elseif ( is_home() ) {
				$page_id = get_option( 'page_for_posts' );
				$title   = get_the_title( absint( $page_id ) );
			} elseif ( is_post_type_archive() ) {
				if ( is_post_type_archive( 'product' ) ) {
					$shop_page_id = get_option( 'woocommerce_shop_page_id' );
					if ( false !== $shop_page_id && ! empty( $shop_page_id ) ) {
						$title = get_the_title( absint( $shop_page_id ) );
					} else {
						$title = post_type_archive_title( '', false );
					}
				} else {
					$title = post_type_archive_title( '', false );
				}
			} elseif ( is_tax() ) {
				$title = single_term_title( '', false );
			} elseif ( is_home() || is_page() ) {
				$title = single_post_title( '', false );
			} elseif ( is_category() || is_tag() ) {
				$title = single_term_title( '', false );
			}

			/*
				This is a feature code if we add filter on the author page
				elseif ( is_author() && get_queried_object() ) {
				$author          eg get_queried_object();
				$title['title'] eq $author->display_name;
			}*/

			return $title;
		}

		/**
		 * Replace url_keys/tags in string with variables
		 *
		 * @param string $rewrite_rule_string
		 *
		 * @return string Replaced url_keys with valiables
		 */
		public static function replace_url_keys_in_rewrite_rule_string( $rewrite_rule_string ) {
			if ( null === self::$page_title ) {
				self::$page_title = self::get_current_page_title();
			}

			$page_title = self::$page_title;
			$site_name  = get_bloginfo( 'name', 'display' );

			$replace_string = $rewrite_rule_string;
			$replace_string = str_replace( self::$separator . 'site_name' . self::$separator, $site_name, $replace_string );
			$replace_string = str_replace( self::$separator . 'page_title' . self::$separator, $page_title, $replace_string );

			$parse_query_vars = $GLOBALS['pwf_main_query']['parse_query_vars'];
			$selected_options = $parse_query_vars->selected_items();

			foreach ( $selected_options as $url_key => $data ) {
				$term_name = '';

				if ( 'priceslider' === $data['field_type'] || 'rangeslider' === $data['field_type'] ) {
					$url_key_min = self::$separator . $url_key . '_min_value' . self::$separator;
					$url_key_max = self::$separator . $url_key . '_max_value' . self::$separator;

					$replace_string = str_replace( $url_key_min, $data['values'][0], $replace_string );
					$replace_string = str_replace( $url_key_max, $data['values'][1], $replace_string );
				} elseif ( 'search' === $data['field_type'] ) {
					$term_name = $data['values'][0];
				} else {
					if ( 'taxonomy' === $data['type'] ) {
						$term = get_term( $data['values'][0], $data['key'] );
						if ( ! is_wp_error( $term ) ) {
							$term_name = $term->name;
						}
					} elseif ( 'meta' === $data['type'] ) {
						$term_name = $data['selected_values'][0]['label'];
					} elseif ( 'on_sale' === $data['type'] ) {
						$term_name = esc_html_x( 'On sale', 'Display with SEO', 'pwf-woo-filter' );
					} elseif ( 'featured' === $data['type'] ) {
						$term_name = esc_html_x( 'Featured', 'Display with SEO', 'pwf-woo-filter' );
					} elseif ( 'vendor' === $data['type'] ) {
						$user_id = $data['values'][0];
						$user    = get_user_by( 'ID', $user_id );
						if ( false !== $user_id ) {
							$term_name = $user->data->display_name;
						}
					} elseif ( 'rating' === $data['type'] ) {
						$term_name = esc_html_x( 'Rate', 'Display with SEO', 'pwf-woo-filter' );
					}
				}

				if ( ! empty( $term_name ) ) {
					$replace_string = str_replace( self::$separator . $url_key . self::$separator, $term_name, $replace_string );
				}
			}

			return $replace_string;
		}

		/**
		 * Add page number to meta title
		 * @param string $title meta title rule
		 *
		 * @return string Title after add page number
		 */
		protected static function add_page_number_to_meta_title( $title ) {
			$current_page_number = Pwf_Hook_Wp_Query::get_current_page_number();
			if ( $current_page_number < 2 ) {
				return $title;
			}

			$position = 'before_site_name';
			/**
			 * @position before_site_name | at_end
			 */
			$position  = apply_filters( 'pwf_seo_page_number_position_in_meta_title', $position );
			$separator = apply_filters( 'pwf_seo_page_number_separator_in_meta_title', '|' );
			if ( ! empty( $separator ) ) {
				$separator .= ' ';
			}

			$page_num_str = $separator . esc_html_x( 'Page', 'Display with SEO', 'pwf-woo-filter' ) . ' ' . $current_page_number;

			if ( 'before_site_name' === $position ) {
				$site_name = self::$separator . 'site_name' . self::$separator;
				if ( strpos( $title, $site_name ) !== false ) {
					$page_num_str = $page_num_str . ' ' . $site_name;
					$title        = str_replace( $site_name, $page_num_str, $title );
				} else {
					$title .= ' ' . $page_num_str;
				}
			} else {
				$title .= ' ' . $page_num_str;
			}

			return $title;
		}

		/**
		 * Replace title tag variable
		 * Default variables page_title, site_name
		 *
		 * @return string
		 */
		public static function get_title_tag_rewrite_rule( $title ) {
			if ( empty( self::$rewrite_rule_data['meta_title'] ) ) {
				return $title;
			}

			$title = self::$rewrite_rule_data['meta_title'];

			$add_page_number = apply_filters( 'pwf_seo_display_page_number_in_meta_title', true );
			if ( $add_page_number ) {
				$title = self::add_page_number_to_meta_title( $title );
			}

			$title = self::replace_url_keys_in_rewrite_rule_string( $title );

			return esc_attr( $title );
		}

		/**
		 * Replace meta description with variables.
		 * @param string $description meta description.
		 *
		 * @return string
		 */
		public static function get_meta_description_rewrite_rule( $description ) {
			if ( empty( self::$rewrite_rule_data['meta_description'] ) ) {
				return $description;
			}

			$description = self::replace_url_keys_in_rewrite_rule_string( self::$rewrite_rule_data['meta_description'] );

			return esc_attr( $description );
		}

		/**
		 * Add Breadcumbs string at the end.
		 *
		 * @return string $breadcumbs
		 */
		public static function get_breadcumbs() {
			$breadcumbs = '';
			if ( ! empty( self::$rewrite_rule_data['breadcumbs'] ) ) {
				$breadcumbs = self::replace_url_keys_in_rewrite_rule_string( self::$rewrite_rule_data['breadcumbs'] );
				$breadcumbs = esc_html( $breadcumbs );
			}

			return $breadcumbs;
		}

		/**
		 * Build canoncial url
		 *
		 * Check is main query or the page is or archive tax or category or tag
		 *
		 * check if the page has page link and add it at the end or global page
		 * check if it clean or pretty link and has prefixed before $url_selected_options
		 * check if the page has orderby
		 *
		 * @param string $url
		 *
		 * @return string $url
		 */
		public static function generate_canoncial_url( $url ) {
			if ( self::is_pretty() ) {
				$pattern = '/page\/[0-9]+\//';
				preg_match_all( $pattern, $url, $matches );
				if ( ! empty( $matches ) & ! empty( $matches[0] ) ) {
					$page_num_str = $matches[0][0];
					$url          = preg_replace( $pattern, '', $url );
				} else {
					$page_num_str = '';
				}

				$prefixed = Pwf_Filter_Manager::get_pretty_urls_prefixed();
				if ( ! empty( $prefixed ) ) {
					$prefixed = sanitize_key( $prefixed ) . '/';
				}

				$url = $url . $prefixed . self::$url_for_selected_options . $page_num_str;

			} else {
				$page_string = '';
				if ( 'page' === self::$current_page_type ) {
					if ( isset( $_GET['product-page'] ) && ! empty( $_GET['product-page'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
						$page_number = absint( $_GET['product-page'] ); //phpcs:ignore WordPress.Security.NonceVerification.Recommended
						if ( $page_number > 1 ) {
							$page_string = 'product-page=' . $page_number . '&';
						}
					} elseif ( isset( $_GET['page'] ) && ! empty( $_GET['page'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
						$page_number = absint( $_GET['page'] ); //phpcs:ignore WordPress.Security.NonceVerification.Recommended
						if ( $page_number > 1 ) {
							$page_string = 'page=' . $page_number . '&';
						}
					}
				}

				$url = $url . '?' . $page_string . self::$clean_url_selected_options;
			}

			return esc_url( $url );
		}

		/**
		 * Require to work with both pretty url and clean url
		 *
		 * Build canonical URL by add selected options and pages
		 * @param string $url current page url
		 *
		 * @return string $url
		 */
		public static function canonical_url( $url ) {
			if ( empty( $url ) ) {
				return $url;
			}

			if ( empty( self::$canoncial_url ) ) {
				self::$canoncial_url = self::generate_canoncial_url( $url );
			}

			return esc_url( self::$canoncial_url );
		}

		public static function get_canonical_url() {
			return esc_url( self::$canoncial_url );
		}

		/**
		 * Return canoncial_url without page Number
		 */
		public static function get_breadcumbs_url() {
			$breadcumbs_url = '';
			if ( ! empty( self::$canoncial_url ) ) {
				$pattern        = '/page\/[0-9]+\//';
				$breadcumbs_url = preg_replace( $pattern, '', self::$canoncial_url );
			}

			return $breadcumbs_url;
		}

		/**
		 * Output Seo description on frontend
		 * @see custom widget and shortcode
		 */
		public static function seo_short_description() {
			if ( ! empty( self::$rewrite_rule_data['short_description'] ) ) {
				$short_desc = self::replace_url_keys_in_rewrite_rule_string( self::$rewrite_rule_data['short_description'] );
				if ( $short_desc ) {
					echo '<div class="pwf-woo-filter-short-description"><p>' . wp_kses_post( $short_desc ) . '</p></div>';
				}
			}
		}

		/**
		 * Add compatible Fro SEO plugins && H1 title
		 */
		public function add_seo_data() {
			if ( defined( 'WPSEO_NAMESPACES' ) ) {
				new Pwf_Yoast_Seo();
			} elseif ( function_exists( 'rank_math' ) ) {
				new Pwf_Rank_Math();
			} elseif ( function_exists( 'aioseo' ) ) {
				new Pwf_All_In_One_Seo();
			} elseif ( defined( '_SQ_PLUGIN_NAME_' ) ) {
				new Pwf_Squirrly_Seo();
			} elseif ( defined( 'SEOPRESS_VERSION' ) ) {
				new Pwf_Seo_Press();
			}

			/**
			 * Modify H1 title
			 */
			$enable_h1_title = apply_filters( 'pwf_seo_enable_to_change_h1_title', true );
			if ( $enable_h1_title ) {
				if ( is_woocommerce() ) {
					$woocommerce_page = apply_filters( 'pwf_seo_h1_change_woocommerce_page_title', true );
					if ( $woocommerce_page ) {
						add_filter( 'woocommerce_page_title', array( $this, 'woocommerce_page_title' ), 10, 1 );
					}
				} else {
					$archive_title = apply_filters( 'pwf_seo_h1_change_archive_title', true );
					if ( $archive_title ) {
						add_filter( 'get_the_archive_title', array( $this, 'modify_the_archive_title' ), 10, 2 );
					}

					$term_title = apply_filters( 'pwf_seo_h1_change_term_title', false );
					if ( $term_title ) {
						add_filter( 'single_cat_title', array( $this, 'modify_term_title' ), 10, 1 );
						add_filter( 'single_tag_title', array( $this, 'modify_term_title' ), 10, 1 );
						add_filter( 'single_term_title', array( $this, 'modify_term_title' ), 10, 1 );
					}
				}

				$the_title = apply_filters( 'pwf_seo_h1_change_the_title', false );
				if ( $the_title ) {
					/**
					 * Not working well
					 * Some times display - possible infinite loop
					 * OR - change Menu item name
					 * For clients- be careful if you enable this filter
					 */
					add_filter( 'the_title', array( $this, 'modify_the_title' ), 10, 2 );
				}
			}

			do_action( 'pwf_seo_after_seo_apply_filters' );
		}

		/**
		 * Modify H1 title on the frontend
		 *
		 * @return string|empty Modify title
		 */
		public static function modify_h1_title() {
			$title = '';
			if ( ! empty( self::$rewrite_rule_data['h1_title'] ) ) {
				$title = self::replace_url_keys_in_rewrite_rule_string( self::$rewrite_rule_data['h1_title'] );
				$title = esc_html( $title );
			}

			return $title;
		}

		/**
		 * Modify WooCommerce Page Title
		 * @see function woocommerce_page_title()
		 */
		public function woocommerce_page_title( $title ) {
			$new_title = self::modify_h1_title();
			if ( ! empty( $new_title ) ) {
				$title = $new_title;
			}

			return $title;
		}

		/**
		 * Modify Archive title
		 * @see function get_the_archive_title()
		 */
		public function modify_the_archive_title( $title, $orginal_title ) {
			$new_title = self::modify_h1_title();
			if ( ! empty( $new_title ) ) {
				$title = $new_title;
			}

			return $title;
		}

		/**
		 * Modift term, category, tag title on the frontend
		 * @see function single_cat_title(), single_tag_title(), single_term_title
		 */
		public function modify_term_title( $title ) {
			$new_title = self::modify_h1_title();
			if ( ! empty( $new_title ) ) {
				$title = $new_title;
			}

			return $title;
		}

		/**
		 * using with apply_filters the_title
		 */
		public function modify_the_title( $title, $id ) {
			if ( absint( self::$current_page_id ) === absint( $id ) ) {
				$new_title = self::modify_h1_title();
				if ( ! empty( $new_title ) ) {
					$title = $new_title;
					$title = esc_html( $title );
				}
				remove_filter( 'the_title', array( $this, 'modify_the_title' ), 10, 2 );
			}

			return $title;
		}

		protected static function is_plugin_activated() {
			return Pwf_Main::is_plugin_activated();
		}
	}
}
