<?php

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'Pwf_Update' ) ) {

	/**
	 * @ since 1.5.4
	 */
	class Pwf_Update {

		public function __construct( array $args ) {
			foreach ( $args as $arg ) {
				switch ( $arg ) {
					case 'update_filter_post':
						self::update_filter_post();
						break;
					case 'add_pretty_urls':
						self::add_pretty_urls();
						break;
					case 'update_filter_to_work_with_any_post_type':
						self::update_filter_to_work_with_any_post_type();
						add_action( 'admin_notices', array( $this, 'admin_notice' ) );
						break;
				}
			}
		}

		protected static function get_query_args() {
			$args = array(
				'post_type'           => 'pwf_woofilter',
				'post_status'         => 'any',
				'ignore_sticky_posts' => -1,
				'posts_per_page'      => -1,
			);

			return $args;
		}

		protected static function update_filter_post() {
			$shop_filter_id = get_option( 'pwf_shop_filter_id', '' );
			$filter_posts   = get_posts( self::get_query_args() );

			if ( $filter_posts ) {
				foreach ( $filter_posts as $post ) {
					$post_id = absint( $post->ID );
					$meta    = get_post_meta( absint( $post_id ), '_pwf_woo_post_filter', true );
					if ( ! empty( $meta ) ) {
						$settings = self::update_filter_settings( $meta['setting'] );
						if ( absint( $shop_filter_id ) === $post_id ) {
							$settings['filter_query_type'] = 'main_query';

							$save_option = array(
								'shop_archive' => $post_id,
							);
							update_option( 'pwf_woo_main_query_filters', $save_option, 'no' );
							delete_option( 'pwf_shop_filter_id' );
						}

						$items = self::update_filter_items( $meta['items'] );

						$meta_data = array(
							'setting' => $settings,
							'items'   => $items,
						);
						update_post_meta( $post_id, '_pwf_woo_post_filter', $meta_data );
					}
				}
			}
		}

		protected static function update_filter_settings( $settings ) {
			$settings['title_toggle_icon']    = $settings['title_toggle_icon'] ?? 'arrow';
			$settings['term_toggle_icon']     = $settings['term_toggle_icon'] ?? 'arrow';
			$settings['filter_query_archive'] = $settings['filter_query_archive'] ?? '';
			$settings['filter_query_pages']   = $settings['filter_query_pages'] ?? '';

			if ( isset( $settings['is_shortcode'] ) && 'on' === $settings['is_shortcode'] ) {
				$settings['filter_query_type'] = 'custom_query';
			} else {
				$settings['filter_query_type'] = '';
			}

			unset( $settings['is_shortcode'] );

			return $settings;
		}

		protected static function update_filter_items( $items ) {
			foreach ( $items as $key => $item ) {
				if ( 'column' !== $item['item_type'] && 'button' !== $item['item_type'] ) {
					if ( ! isset( $item['depends_on'] ) ) {
						$items[ $key ]['depends_on'] = '';
					}
					if ( ! isset( $item['depends_on_operator'] ) ) {
						$items[ $key ]['depends_on_operator'] = 'and';
					}
					if ( ! isset( $item['product_variations'] ) ) {
						if ( in_array( $item['item_type'], array( 'checkboxlist', 'radiolist', 'textlist', 'dropdownlist', 'colorlist', 'boxlist' ), true ) ) {
							$items[ $key ]['product_variations'] = '';
						}
					}
					if ( ! isset( $item['dropdown_style'] ) && 'dropdownlist' === $item['item_type'] ) {
						$items[ $key ]['dropdown_style'] = 'default';
					}
					if ( ! isset( $item['search_inside'] ) && 'search' === $item['item_type'] ) {
						$items[ $key ]['search_inside'] = 'all';
					}
				}
			}

			return $items;
		}

		/**
		 * @since 1.5.7
		 */
		protected static function add_pretty_urls() {
			$filter_posts = get_posts( self::get_query_args() );
			if ( $filter_posts ) {
				foreach ( $filter_posts as $post ) {
					$post_id = absint( $post->ID );
					$meta    = get_post_meta( absint( $post_id ), '_pwf_woo_post_filter', true );
					if ( ! empty( $meta ) ) {
						$meta['setting']['pretty_urls'] = 'off';

						$meta_data = array(
							'setting' => $meta['setting'],
							'items'   => $meta['items'],
						);
						update_post_meta( $post_id, '_pwf_woo_post_filter', $meta_data );
					}
				}

				Pwf_Routes_Manager::regenerates_routes();
			}
		}

		/**
		 * Update PWF filter to work with any post type
		 *
		 * @since 1.6.9
		 *
		 * @return void
		 */
		public static function update_filter_to_work_with_any_post_type() {
			$filter_posts = get_posts( self::get_query_args() );
			if ( $filter_posts ) {
				foreach ( $filter_posts as $post ) {
					$post_id = absint( $post->ID );
					$meta    = get_post_meta( absint( $post_id ), '_pwf_woo_post_filter', true );
					if ( ! empty( $meta ) ) {
						$meta['setting']['post_type'] = 'product';

						$display_pages = array();
						if ( ! empty( $meta['setting']['filter_query_archive'] ) ) {
							$filter_query_archive = $meta['setting']['filter_query_archive'];
							foreach ( $filter_query_archive as $term ) {
								if ( 'shop_archive' === $term ) {
									array_push( $display_pages, 'product__archive' );
								} else {
									$split = explode( '__', $term );
									if ( ! isset( $split[2] ) ) {
										array_push( $display_pages, 'product__' . $term );
									}
								}
							}
						}

						if ( ! empty( $meta['setting']['filter_query_pages'] ) ) {
							$filter_query_pages = $meta['setting']['filter_query_pages'];
							foreach ( $filter_query_pages as $term ) {
								array_push( $display_pages, 'page__' . $term );
							}
						}

						$meta['setting']['filter_display_pages'] = $display_pages;
						unset( $meta['setting']['filter_query_pages'] );
						unset( $meta['setting']['filter_query_archive'] );

						update_post_meta( $post_id, '_pwf_woo_post_filter', $meta );
					}
				}
			}

			// Update options
			$query_filters     = array();
			$filter_option_ids = get_option( 'pwf_woo_query_filters' );
			if ( false === $filter_option_ids ) {
				$main_query_filters   = get_option( 'pwf_woo_main_query_filters', '' );
				$custom_query_filters = get_option( 'pwf_woo_custom_query_filters', '' );

				if ( ! empty( $main_query_filters ) ) {
					foreach ( $main_query_filters as $term => $filter_id ) {
						if ( 'shop_archive' === $term ) {
							$query_filters['product__archive'] = $filter_id;
						} else {
							$query_filters[ 'product__' . $term ] = $filter_id;
						}
					}
				}

				if ( ! empty( $custom_query_filters ) ) {
					foreach ( $custom_query_filters as $term => $filter_id ) {
						$query_filters[ 'page__' . $term ] = $filter_id;
					}
				}

				update_option( 'pwf_woo_query_filters', $query_filters, 'yes' );
				delete_option( 'pwf_woo_main_query_filters' );
				delete_option( 'pwf_woo_custom_query_filters' );
			}
		}

		/**
		 * Update analytics DB table
		 *
		 * @since 1.6.9
		 *
		 * @return void
		 */
		public static function update_db_analytics_table() {
			global $wpdb;

			$is_column_exist = $wpdb->get_results(
				$wpdb->prepare(
					'SELECT COLUMN_NAME FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_NAME = %s AND COLUMN_NAME = %s',
					$wpdb->prefix . 'wc_pwf_filters',
					'post_type'
				)
			);

			if ( empty( $is_column_exist ) ) {
				$wpdb->query( "ALTER TABLE {$wpdb->prefix}wc_pwf_filters ADD post_type VARCHAR(255) NOT NULL DEFAULT 'product' AFTER query_string" );
				update_option( 'pwf_woocommerce_db_version', PWF_WOO_FILTER_DB_VERSION, false );

				$wpdb->update(
					$wpdb->prefix . 'wc_pwf_filters',
					'product_archive',
					array( 'page_type' => 'shop' )
				);
			}
		}

		public function admin_notice() {
			echo '
			<div class="notice notice-info is-dismissible">
				<h2>PWF Woocommerce Product Filters plugin</h2>
				<ul>
					<li>This version has a major update and important changes.</li>
					<li><strong>Support any custom post type</strong>.</li>
					<li><a href="https://docs.mostafaa.net/plugins/pwfwoofilter/tuts.html#general" target="_blank">read more</a>.</li>
				</ul>
			</div>';
		}
	}
}
