<?php

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'Pwf_Seo_Settings_Page' ) ) {
	/**
	 * Seo settings page
	 *
	 * @since 1.7.7
	 */
	class Pwf_Seo_Settings_Page {

		/**
		 * Option name
		 */
		protected static $option_name = 'pwf_woo_filter_seo_rules_options'; // pwf_woo_filter_seo_rules

		protected static $where_pages;

		/**
		 * Hold filter posts ID these integrate with page rules
		 */
		protected static $filter_data_ids = array();

		public function __construct() {
			self::init();
			add_action( 'admin_footer', array( $this, 'custom_enqueue_script' ), 1000 );
		}

		/**
		 * Fields for each rewrite rule
		 *
		 * @return array
		 */
		protected static function default_fields() {
			$fields = array(
				array(
					'type'      => 'text',
					'name'      => 'rewrite_rule',
					'title'     => esc_html__( 'SEO URL', 'pwf-woo-filter' ),
					'desc'      => esc_html__( 'SEO URL rewrite rule. The last part of the URL.', 'pwf-woo-filter' ),
					'value'     => '',
					'css_class' => 'rewrite-rule',
					'button'    => array(
						'icon'      => 'insert',
						'css_class' => 'insert-rule',
					),
				),
				array(
					'type'      => 'text',
					'name'      => 'meta_title',
					'title'     => esc_html__( 'Meta Title', 'pwf-woo-filter' ),
					'desc'      => esc_html__( 'Display between', 'pwf-woo-filter' ) . ' &lt;title&gt;Title&lt;/title&gt;',
					'value'     => '',
					'css_class' => 'meta-title',
					'button'    => array(
						'icon'      => 'update',
						'css_class' => 'update-rule',
					),
				),
				array(
					'type'      => 'textarea',
					'name'      => 'meta_description',
					'title'     => esc_html__( 'Meta description', 'pwf-woo-filter' ),
					'desc'      => '',
					'value'     => '',
					'css_class' => 'meta-description',
					'button'    => array(
						'icon'      => 'update',
						'css_class' => 'update-rule',
					),
				),
				array(
					'type'      => 'text',
					'name'      => 'h1_title',
					'title'     => esc_html__( 'Page Title H1', 'pwf-woo-filter' ),
					'desc'      => '',
					'value'     => '',
					'css_class' => 'h1-title',
					'button'    => array(
						'icon'      => 'update',
						'css_class' => 'update-rule',
					),
				),
				array(
					'type'      => 'text',
					'name'      => 'breadcumbs',
					'title'     => esc_html__( 'Breadcumbs', 'pwf-woo-filter' ),
					'desc'      => '',
					'value'     => '',
					'css_class' => 'breadcumbs',
					'button'    => array(
						'icon'      => 'update',
						'css_class' => 'update-breadcumbs',
					),
				),
				array(
					'type'      => 'textarea',
					'name'      => 'short_description',
					'title'     => esc_html__( 'Short description', 'pwf-woo-filter' ),
					'desc'      => esc_html__( 'Use a shortcode or a widget to display this field on the frontend', 'pwf-woo-filter' ),
					'value'     => '',
					'css_class' => 'short-description',
					'button'    => array(
						'icon'      => 'update',
						'css_class' => 'update-rule',
					),
				),
			);

			return $fields;
		}

		protected static function init() {
			self::render_page();
		}

		/**
		 * Get page arguments
		 * When the user select page from page panel settings
		 *
		 * @return array page_type tax|page|archive, page_id post_type| page_id | term_id
		 */
		protected static function get_page_args( $page_type ) {
			$args = array(
				'page_type' => '', // taxonomy/archive/page
				'post_type' => '',
				'page_id'   => '',
				'taxonomy'  => '',
				'tax_id'    => '',
			);

			$split = explode( '__', $page_type );
			if ( 2 === count( $split ) ) {
				if ( 'archive' === $split[1] ) {
					$args['page_type'] = 'archive';
					$args['post_type'] = $split[0];
				} else {
					$args['page_type'] = 'page';
					$args['page_id']   = $split[1];
				}
			} else {
				$args['page_type'] = 'taxonomy';
				$args['post_type'] = $split[0];
				$args['taxonomy']  = $split[1];
				$args['term_id']   = $split[2];
			}

			return $args;
		}

		/**
		 * Get page Title
		 *
		 * @param $args array
		 *
		 * @return string The Page Title
		 */
		protected static function get_page_title( $args ) {
			$title = '';
			if ( 'archive' === $args['page_type'] ) {
				$post_obj = get_post_type_object( $args['post_type'] );
				if ( 'product' === $post_obj->name ) {
					$title = 'Shop';
				} elseif ( ! empty( $post_obj->has_archive ) ) {
					$title = $post_obj->has_archive;
				} else {
					$title = $post_obj->labels->singular_name;
				}
				if ( ! empty( $title ) ) {
					$title .= ' ' . esc_html__( 'Archive', 'pwf-woo-filter' );
				}
			} elseif ( 'page' === $args['page_type'] ) {
				$title = get_the_title( $args['page_id'] );
			} elseif ( 'taxonomy' === $args['page_type'] ) {
				if ( 'all' !== $args['term_id'] ) {
					$term = get_term(
						absint( $args['term_id'] ),
						esc_attr( $args['taxonomy'] ),
					);
					if ( ! is_wp_error( $term ) ) {
						$title = $term->name;
					}
				} else {
					$taxonomy = get_taxonomy( $args['taxonomy'] );
					if ( false !== $taxonomy ) {
						$title = esc_html__( 'All', 'pwf-woo-filter' ) . ' ' . $taxonomy->labels->name;
					}
				}
			}

			return $title;
		}

		/**
		 * Get page link on the frontend
		 * Used with the preview button
		 *
		 * @param $args array
		 *
		 * @return string The url
		 */
		protected static function get_page_url( $args ) {
			$url = '';
			if ( 'archive' === $args['page_type'] ) {
				$url = get_post_type_archive_link( $args['post_type'] );
			} elseif ( 'page' === $args['page_type'] ) {
				$url = get_page_link( absint( $args['page_id'] ) );
			} elseif ( 'taxonomy' === $args['page_type'] ) {
				if ( 'all' !== $args['term_id'] ) {
					$url = get_term_link( absint( $args['term_id'] ), esc_attr( $args['taxonomy'] ) );
				} else {
					// return one link
					$terms = get_terms(
						esc_attr( $args['taxonomy'] ),
						array(
							'hide_empty' => false,
							'fields'     => 'ids',
						)
					);
					if ( ! is_wp_error( $terms ) ) {
						$term_id = $terms[0];
						$url     = get_term_link( absint( $term_id ), esc_attr( $args['taxonomy'] ) );
					}
				}
			}

			return $url;
		}

		/**
		 * Get filter ID assigned to the current page
		 *
		 * @param string $page_type archive|taxonomy|page
		 *
		 * @return int|Empty The filter ID
		 */
		protected static function get_filter_id( $page_type ) {
			$filter_id  = '';
			$filter_ids = get_option( 'pwf_woo_query_filters', '' );

			if ( ! empty( $filter_ids ) && is_array( $filter_ids ) ) {
				if ( isset( $filter_ids[ $page_type ] ) ) {
					$filter_id = $filter_ids[ $page_type ];
				} else {
					// Search for taxonomy name or post archive
					$split = explode( '__', $page_type );
					if ( isset( $split[2] ) ) {
						if ( 'all' === $split[2] ) {
							$archive_page = $split[0] . '__archive';
							if ( isset( $filter_ids[ $archive_page ] ) ) {
								$filter_id = $filter_ids[ $archive_page ];
							}
						} else {
							$tax_all = $split[0] . '__' . $split[1] . '__all';
							if ( isset( $filter_ids[ $tax_all ] ) ) {
								$filter_id = $filter_ids[ $tax_all ];
							} else {
								$archive_page = $split[0] . '__archive';
								if ( isset( $filter_ids[ $archive_page ] ) ) {
									$filter_id = $filter_ids[ $archive_page ];
								}
							}
						}
					}
				}
			}

			return $filter_id;
		}

		/**
		 * Build filter data for ajax
		 * @param int The filter ID
		 *
		 * @return array The filter url_keys and values for SEO.
		 */
		protected static function get_filter_data( $filter_id ) {
			// You are here
			$result      = array();
			$url_keys    = array();
			$filter_id   = Pwf_Filter_Manager::get_filter_id( $filter_id );
			$filter      = Pwf_Filter_Manager::get_filter_settings_and_items( $filter_id );
			$filed_types = array( 'checkboxlist', 'radiolist', 'dropdownlist', 'colorlist', 'boxlist', 'textlist' );

			if ( ! empty( $filter ) && isset( $filter['items'] ) ) {
				$result['filter_id'] = $filter_id;

				$filter_items = Pwf_Filter_Manager::get_filter_items_without_columns( $filter['items'] );
				foreach ( $filter_items as $item ) {
					$data = array();
					if ( in_array( $item['item_type'], $filed_types, true ) ) {
						if ( in_array( $item['source_of_options'], array( 'category', 'tag', 'attribute', 'taxonomy' ), true ) ) {
							if ( 'category' === $item['source_of_options'] ) {
								$taxonomy_name = 'product_cat';
							} elseif ( 'tag' === $item['source_of_options'] ) {
								$taxonomy_name = 'product_tag';
							} elseif ( 'attribute' === $item['source_of_options'] ) {
								$taxonomy_name = $item['item_source_attribute'];
							} elseif ( 'taxonomy' === $item['source_of_options'] ) {
								$taxonomy_name = $item['item_source_taxonomy'];
							}

							$terms = get_terms(
								array(
									'taxonomy'   => $taxonomy_name,
									'hide_empty' => true,
								)
							);
							if ( ! is_wp_error( $terms ) ) {
								foreach ( $terms as $term ) {
									$data[] = array(
										'slug'  => esc_attr( $term->slug ),
										'label' => esc_attr( $term->name ),
									);
								}
							}

							$result['url_keys'][ $item['url_key'] ] = array(
								'field_type' => $item['item_type'],
								'source'     => 'taxonomy',
								'values'     => $data,
							);
						} else {
							if ( 'meta' === $item['source_of_options'] ) {
								$metafield = apply_filters( 'pwf_filter_item_meta_field_variables', $item['metafield'], $item['meta_key'], $filter_id );
								if ( ! empty( $metafield ) ) {
									foreach ( $metafield as $meta ) {
										$data[] = array(
											'slug'  => esc_attr( $meta['slug'] ),
											'label' => esc_attr( $meta['label'] ),
										);
									}

									$result['url_keys'][ $item['url_key'] ] = array(
										'field_type' => $item['item_type'],
										'source'     => 'meta',
										'values'     => $data,
									);
								}
							} elseif ( 'author' === $item['source_of_options'] ) {
								$args = array(
									'hide_empty' => false,
									'fields'     => array( 'ID', 'display_name', 'user_nicename' ),
								);
								if ( ! empty( $item['user_roles'] ) ) {
									$args['role__in'] = array_map( 'esc_attr', $item['user_roles'] );
								}
								if ( ! empty( $item['include'] ) ) {
									$args['include'] = array_map( 'absint', $item['include'] );
								} elseif ( ! empty( $item['exclude'] ) ) {
									$args['exclude'] = array_map( 'absint', $item['exclude'] );
								}

								$get_users = get_users( $args );
								if ( ! empty( $get_users ) ) {
									foreach ( $get_users as $user ) {
										$data[] = array(
											'slug'  => esc_attr( $user->user_nicename ),
											'label' => esc_attr( $user->display_name ),
										);
									}

									$result['url_keys'][ $item['url_key'] ] = array(
										'field_type' => $item['item_type'],
										'source'     => 'author',
										'values'     => $data,
									);
								}
							} elseif ( 'on_sale' === $item['source_of_options'] ) {
								$result['url_keys'][ $item['url_key'] ] = array(
									'field_type' => $item['item_type'],
									'source'     => 'on_sale',
									'values'     => array(
										array(
											'slug'  => 'yes',
											'label' => esc_html__( 'On sale', 'pwf-woo-filter' ),
										),
									),
								);
							} elseif ( 'featured' === $item['source_of_options'] ) {
								$result['url_keys'][ $item['url_key'] ] = array(
									'field_type' => $item['item_type'],
									'source'     => 'featured',
									'values'     => array(
										array(
											'slug'  => 'yes',
											'label' => esc_html__( 'Featured', 'pwf-woo-filter' ),
										),
									),
								);
							}
						}
					} else {
						// should be rating, search, priceslider, range slider, date
						if ( 'priceslider' === $item['item_type'] || 'rangeslider' === $item['item_type'] ) {
							$result['url_keys'][ $item['url_key'] ] = array(
								'field_type' => $item['item_type'],
								'source'     => $item['item_type'],
								'values'     => array(),
							);
						} elseif ( 'rating' === $item['item_type'] ) {
							$rating    = array();
							$rate_text = esc_html__( 'rating', 'pwf-woo-filter' );
							$from_text = esc_html__( 'from', 'pwf-woo-filter' );
							$to_text   = esc_html__( 'to', 'pwf-woo-filter' );

							if ( 'on' === $item['up_text'] ) {
								for ( $index = 1; $index <= 4; $index++ ) {
									$rating[] = array(
										'slug'  => $index . '-5',
										'label' => $rate_text . ' ' . esc_html__( 'from', 'pwf-woo-filter' ) . ' ' . $index . ' ' . $to_text . ' 5',
									);
								}
							} else {
								for ( $index = 1; $index <= 5; $index++ ) {
									$rating[] = array(
										'slug'  => $index,
										'label' => $rate_text . ' ' . $index,
									);
								}
							}

							$result['url_keys'][ $item['url_key'] ] = array(
								'field_type' => $item['item_type'],
								'source'     => 'rating',
								'values'     => $rating,
								'up_text'    => $item['up_text'],
							);
						} elseif ( 'search' === $item['item_type'] ) {
							$result['url_keys'][ $item['url_key'] ] = array(
								'field_type' => $item['item_type'],
								'source'     => 'search',
								'values'     => array(
									array(
										'slug'  => 'search_text',
										'label' => 'search_text_text',
									),
								),
							);
						}
					}
				}
			}

			return $result;
		}

		/**
		 * Get each filter data used with SEO
		 *
		 * @return array
		 */
		protected static function get_filters_data() {
			$filers_data = array();
			if ( ! empty( self::$filter_data_ids ) ) {
				foreach ( self::$filter_data_ids as $filter_id ) {
					$data = self::get_filter_data( $filter_id );
					if ( ! empty( $data ) ) {
						array_push( $filers_data, $data );
					}
				}
			}

			return $filers_data;
		}

		protected static function render_page() {
			if ( ! current_user_can( 'manage_options' ) ) {
				return;
			}

			Pwf_Seo_Templates::start_render_page();
			self::generate_html_rules();
			echo '</div>';
			echo '<div class="wrap-add-new-page-rule">';
			Pwf_Seo_Templates::add_page_rule_link();
			Pwf_Seo_Templates::end_render_page();
		}

		/**
		 * Generate SEO Rules if exists
		 */
		protected static function generate_html_rules() {
			$rules = get_option( self::$option_name );
			if ( ! empty( $rules ) ) {
				foreach ( $rules as $index => $rule ) {
					self::print_page_rule_content( $index, $rule );
				}
			}
		}

		/**
		 * The page rule contain group of rules and settings
		 *
		 */
		protected static function print_page_rule_content( $index, $args ) {
			$filter_id = self::get_filter_id( $args['page_type'] );
			if ( ! empty( $filter_id ) && ! in_array( $filter_id, self::$filter_data_ids, true ) ) {
				array_push( self::$filter_data_ids, $filter_id );
			}

			$page_args = self::get_page_args( $args['page_type'] );
			$title     = self::get_page_title( $page_args );
			$url       = self::get_page_url( $page_args );

			echo '<div id="single-page-rule-content-' . absint( $index ) . '" class="single-page-rule-content postbox closed"';
			echo ' data-filter-id="' . absint( $filter_id ) . '" data-page-type="' . esc_attr( $args['page_type'] ) . '" data-page-url="' . esc_url( $url ) . '">';
			Pwf_Seo_Templates::panel_header( $title, 'page-panel-header', 'page_rule' );

			echo '<div class="inside">';

			$pwf_databas       = new Pwf_Database_Query();
			$pages             = $pwf_databas->filter_query_pages( 'post', true );
			self::$where_pages = $pages;
			Pwf_Seo_Templates::pages_dropdown_menu( $args['page_type'], $pages );
			Pwf_Seo_Templates::language_template( self::get_available_languages(), $args );

			echo '<div class="pwf-page-rule-error-message"></div>';
			echo '<div class="inside-content group-page-rules">';
			if ( isset( $args['rules'] ) ) {
				foreach ( $args['rules'] as $rule ) {
					Pwf_Seo_Templates::rule_content( $rule, self::default_fields() );
				}
			}
			echo '</div>';

			Pwf_Seo_Templates::add_new_seo_rule_button();

			echo '</div>';
			echo '</div>';
		}

		public static function get_page_data() {
			check_admin_referer( 'ajax_pro_woo_seo_filter_nonce', 'nonce' );

			if ( empty( $_POST['page_type'] ) ) {
				$send_data = array(
					'success' => 'false',
					'message' => esc_html__( 'There is no selected page to get', 'pwf-woo-filter' ),
				);
			} else {
				$page_type   = esc_attr( $_POST['page_type'] );
				$filter_id   = self::get_filter_id( $page_type );
				$filter_data = self::get_filter_data( $filter_id );
				$page_url    = self::get_page_url( self::get_page_args( $page_type ) );

				$send_data = array(
					'success'     => 'true',
					'filter_id'   => $filter_id,
					'filter_data' => $filter_data,
					'page_url'    => $page_url,
				);
			}

			echo wp_json_encode( $send_data );
			wp_die();
		}

		public static function save_seo_settings() {
			check_admin_referer( 'ajax_pro_woo_seo_filter_nonce', 'nonce' );

			$send_data = array(
				'success' => 'false',
				'message' => '',
				'data'    => '',
			);

			if ( empty( $_POST['data'] ) ) {
				$send_data = array(
					'success' => 'false',
					'message' => esc_html__( 'There is no data to save', 'pwf-woo-filter' ),
				);
			} else {
				$seo_rules  = array();
				$page_rules = json_decode( stripslashes( $_POST['data'] ), true );
				foreach ( $page_rules as $page_rule ) {
					$current              = array();
					$current['page_type'] = sanitize_key( $page_rule['page_type'] );
					$current['lang']      = sanitize_key( $page_rule['lang'] );
					$current['rules']     = array();
					foreach ( $page_rule['rules'] as $rule ) {
						// using sanitize_text_field() doesn't work well
						$current['rules'][] = array(
							'rewrite_rule'      => ( ! empty( $rule['rewrite_rule'] ) ) ? wp_kses_post( $rule['rewrite_rule'] ) : '',
							'h1_title'          => ( ! empty( $rule['h1_title'] ) ) ? wp_kses_post( $rule['h1_title'] ) : '',
							'meta_title'        => ( ! empty( $rule['meta_title'] ) ) ? wp_kses_post( $rule['meta_title'] ) : '',
							'meta_description'  => ( ! empty( $rule['meta_description'] ) ) ? wp_kses_post( $rule['meta_description'] ) : '',
							'breadcumbs'        => ( ! empty( $rule['breadcumbs'] ) ) ? wp_kses_post( $rule['breadcumbs'] ) : '',
							'short_description' => ( ! empty( $rule['short_description'] ) ) ? wp_kses_post( $rule['short_description'] ) : '',
						);
					}

					array_push( $seo_rules, $current );
				}

				update_option( self::$option_name, $seo_rules );
				$send_data = array(
					'success' => 'true',
					'message' => esc_html__( 'Data saved', 'pwf-woo-filter' ),
				);
			}

			echo wp_json_encode( $send_data );
			wp_die();
		}

		public static function enqueue_styles() {
			$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
			wp_enqueue_style( 'select2', PWF_WOO_FILTER_URI . '/assets/select2/css/select2' . $suffix . '.css', '', '4.0.12' );
			wp_enqueue_style( 'prowoofilteradmin', PWF_WOO_FILTER_URI . '/assets/css/admin/admin' . $suffix . '.css', '', PWF_WOO_FILTER_VER );
			wp_enqueue_style( 'pwf-admin-seo-settings-page', PWF_WOO_FILTER_URI . '/assets/css/admin/seo-settings' . $suffix . '.css', '', PWF_WOO_FILTER_VER );
		}

		public function custom_enqueue_script() {
			$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
			wp_enqueue_script( 'jquery-ui-sortable' );
			wp_enqueue_script( 'select2', PWF_WOO_FILTER_URI . '/assets/select2/js/select2.full.js', '', '4.0.12', true );
			wp_enqueue_script( 'seo-settings', PWF_WOO_FILTER_URI . '/assets/js/admin/seo-settings' . $suffix . '.js', '', PWF_WOO_FILTER_VER, true );

			if ( null === self::$where_pages ) {
				$pwf_databas       = new Pwf_Database_Query();
				self::$where_pages = $pwf_databas->filter_query_pages( 'post', false );
			}

			$filters_data = self::get_filters_data();

			$localize_args = array(
				'nonce'            => wp_create_nonce( 'ajax_pro_woo_seo_filter_nonce' ),
				'pages'            => self::$where_pages,
				'default_fields'   => self::default_fields(),
				'filters_data'     => $filters_data,
				'translated_text'  => self::get_translated_text(),
				'language_options' => self::get_available_languages(),
				'current_language' => self::get_current_language(),
			);

			wp_localize_script( 'seo-settings', 'pro_woo_filter_seo_settings', $localize_args );
		}

		protected static function get_translated_text() {
			$translated = array(
				'page'           => esc_html__( 'Page', 'pwf-woo-filter' ),
				'add_new_rule'   => esc_html__( 'Add New SEO Rule', 'pwf-woo-filter' ),
				'clear'          => esc_html__( 'Clear', 'pwf-woo-filter' ),
				'generate'       => esc_html__( 'Generate', 'pwf-woo-filter' ),
				'variables'      => esc_html__( 'Available variables', 'pwf-woo-filter' ),
				'select_page'    => esc_html__( 'Please, Select Page from dropdown menu', 'pwf-woo-filter' ),
				'delete_rule'    => esc_html__( 'Do you want to delete this item?', 'pwf-woo-filter' ),
				'delete_input'   => esc_html__( 'Do you want to delete the text?', 'pwf-woo-filter' ),
				'empty_rule_url' => esc_html__( 'The input field URL rule is empty. Please, add some data to it.', 'pwf-woo-filter' ),
				'var_not_exists' => esc_html__( 'These variables do not exist on URL rewrite rule.', 'pwf-woo-filter' ),
				'preview_links'  => esc_html__( 'Preview links', 'pwf-woo-filter' ),
				'delete'         => esc_html__( 'Delete', 'pwf-woo-filter' ),
				'no_feature'     => esc_html__( 'This feature has not been added yet.', 'pwf-woo-filter' ),
				'selected_error' => esc_html__( 'The page you have selected does not have the filter assigned to it before.', 'pwf-woo-filter' ),
				'language'       => esc_html__( 'Language', 'pwf-woo-filter' ),
				'select_page'    => esc_html__( 'Select the page that you want to add new rewrite rules at the end of its URL', 'pwf-woo-filter' ),
			);
			return $translated;
		}

		/**
		 * Check if the site is Multi language
		 * check if WPML exist
		 * this field on saved seo option Important. If we are planing to generate site map
		 *
		 * @return string current language
		 */
		protected static function get_current_language() {
			$language = '';
			if ( Pwf_Wpml::is_wpml_defined() ) {
				$language = ICL_LANGUAGE_CODE;
			}

			return $language;
		}

		/**
		 * Get available Languages
		 *
		 * @return array
		 */
		public static function get_available_languages() {
			$results   = array();
			$results[] = array(
				'id'   => '',
				'text' => '',
			);

			if ( Pwf_Wpml::is_wpml_defined() ) {
				$active_languages = apply_filters( 'wpml_active_languages', null );
				foreach ( $active_languages as $lang_code => $lang_data ) {
					$results[] = array(
						'id'   => esc_attr( $lang_code ),
						'text' => esc_attr( $lang_data['native_name'] ),
					);
				}
			}
			return $results;
		}
	}
}
