<?php

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'Pwf_Admin_Settings_Page' ) ) {

	class Pwf_Admin_Settings_Page {

		public static function register() {
			$plugin = new self();
			add_action( 'init', array( $plugin, 'init' ) );
		}

		public function init() {
			add_action( 'admin_menu', array( $this, 'add_settings_page' ) );
			add_action( 'admin_init', array( $this, 'setup_sections' ) );
			add_action( 'admin_init', array( $this, 'setup_fields' ) );
			add_action( 'admin_init', array( $this, 'notice_dismissed' ), 10 );
			add_action( 'admin_notices', array( $this, 'admin_notice_license_activation' ), 10 );

			$validate_options = array(
				'pwf_shop_pretty_urls_prefixed',
				'pwf_transient_time',
				'envato_purchase_code_28181010',
				'pwf_shop_enable_pretty_links',
				'pwf_shop_theme_compitable',
				'pwf_shop_analytics',
				'pwf_shop_analytics_save_user_id',
			);

			foreach ( $validate_options as $option ) {
				add_filter( 'sanitize_option_' . $option, array( $this, 'sanitize_fields' ), 10, 3 );
			}
		}

		/**
		 * Define Enable|Disable
		 */
		protected static function get_enable_disable_options() {
			return array(
				'enable'  => esc_html__( 'Enable', 'pwf-woo-filter' ),
				'disable' => esc_html__( 'Disable', 'pwf-woo-filter' ),
			);
		}

		/**
		 * Verfy purchase code
		 *
		 * @param string $code purchase code
		 * @param string $action delete|update|Empty
		 * @since 1.6.6
		 */
		public static function verify_code( $code, $action = '' ) {
			$end_point  = 'https://verifysales.mostafaa.net/wp-json/evanto-verify-sale/v1/checkpurchasekey/';
			$end_point .= '?code=' . sanitize_key( $code );
			$end_point .= '&site_url=' . Pwf_Main::get_site_url();
			$end_point .= '&item_slug=pwfwoo';

			if ( ! empty( $action ) ) {
				$end_point .= '&action=' . $action;
			}

			$args = array(
				'timeout' => 30,
				'headers' => array(
					'Content-Type' => 'application/json',
				),
			);

			$response      = wp_remote_get( $end_point, $args );
			$response_code = absint( wp_remote_retrieve_response_code( $response ) );
			if ( is_wp_error( $response ) ) {
				if ( isset( $response->errors ) ) {
					$error = $response->errors;
					if ( isset( $error['http_request_failed'] ) ) {
						if ( is_array( $error['http_request_failed'] ) ) {
							$message = $error['http_request_failed'][0];
						} else {
							$message = $error['http_request_failed'];
						}
						$body = array(
							'code'    => 0,
							'message' => 'PWF the verify error code is: ' . $message,
						);
					} else {
						$body = array(
							'code'    => 0,
							'message' => 'PWF: unknown Error',
						);
					}
				} else {
					$body = array(
						'code'    => 0,
						'message' => 'PWF: unknown Error',
					);
				}
			} else {
				if ( 500 === $response_code ) {
					$body = array(
						'code'    => 500,
						'message' => 'Our server is busy, please try again later',
					);
				} else {
					$body = json_decode( wp_remote_retrieve_body( $response ), true );
				}
			}

			return $body;
		}

		/**
		 * Use to verify and delete the purchase code
		 *
		 * @since 1.8.0
		 */
		public static function ajax_settings_page() {
			check_admin_referer( 'pwf_woo_settings_nonce', 'nonce' );

			$action = isset( $_POST['data'] ) ? sanitize_key( $_POST['data'] ) : '';
			if ( empty( $action ) || ! in_array( $action, array( 'delete', 'verify' ), true ) ) {
				$send_data = array(
					'success' => 'error',
					'message' => esc_html__( 'There is no action assign', 'pwf-woo-filter' ),
				);
			} else {
				if ( 'verify' === $action ) {
					$purchase_code = isset( $_POST['purchase_code'] ) ? sanitize_key( $_POST['purchase_code'] ) : '';
					if ( empty( $purchase_code ) ) {
						$send_data = array(
							'success' => 'error',
							'message' => esc_html__( 'Please, add the purchase code.', 'pwf-woo-filter' ),
						);
					} else {
						$results = self::verify_code( $purchase_code, $action );
						if ( 200 === $results['code'] ) {
							$message_type = 'success';
							Pwf_Admin_Main::set_purchase_code( $purchase_code );
						} else {
							$message_type = 'error';
						}

						$send_data = array(
							'message_type' => $message_type,
							'message'      => $results['message'],
						);
					}
				} elseif ( 'delete' === $action ) {
					$saved_purchase_code = Pwf_Main::get_purchase_code();
					$results             = self::verify_code( $saved_purchase_code, $action );
					if ( 200 === absint( $results['code'] ) ) {
						$message_type = 'success';
					} else {
						$message_type = 'error';
					}
					$message_code = isset( $results['message_code'] ) ? absint( $results['message_code'] ) : 0;
					if ( 1 === $message_code || 4 === $message_code ) {
						Pwf_Admin_Main::set_purchase_code( '' );
					}

					$send_data = array(
						'message_type' => $message_type,
						'message'      => $results['message'],
					);
				}
			}

			echo wp_json_encode( $send_data );
			wp_die();
		}

		/**
		 * @param string $message
		 * @param string $type error|success|warning|info
		 */
		protected static function add_settings_error( $message, $id, $type = 'error' ) {
			add_settings_error(
				'pwf_filter_fields',
				$id,
				$message,
				$type
			);
		}

		/**
		 * Create settings page
		 *
		 * @since 1.6.6
		 */
		public function add_settings_page() {
			$submenu = add_submenu_page(
				'edit.php?post_type=pwf_woofilter',
				'Filters Settings',
				'Settings',
				'manage_options',
				'pwf-settings',
				array( $this, 'render_settings_page' ),
				10
			);

			add_action( 'admin_print_styles-' . $submenu, array( $this, 'enqueue_styles' ) );
			add_action( 'admin_print_scripts-' . $submenu, array( $this, 'enqueue_scripts' ) );
		}

		/**
		 * @since 1.6.6
		 */
		public function enqueue_styles() {
			$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
			wp_enqueue_style( 'prowoofilteradmin-settings', PWF_WOO_FILTER_URI . '/assets/css/admin/settings' . $suffix . '.css', '', PWF_WOO_FILTER_VER );
		}

		/**
		 * @since 1.8.0
		 */
		public function enqueue_scripts() {
			wp_enqueue_script( 'pwf-woo-settings', PWF_WOO_FILTER_URI . '/assets/js/admin/settings-page.js', array( 'jquery' ), PWF_WOO_FILTER_VER, true );

			$localize_args = array(
				'nonce' => wp_create_nonce( 'pwf_woo_settings_nonce' ),
			);

			wp_localize_script( 'pwf-woo-settings', 'pwf_woo_settings_page', $localize_args );
		}

		public function render_settings_page() {
			// check user capabilities
			if ( ! current_user_can( 'manage_options' ) ) {
				return;
			}

			self::check_buttons();

			settings_errors();
			?>
				<div class="wrap pwf-settings-page">
					<h2 id="pwf_filter_fields">PWF Settings</h2>
					<form method="POST" action="options.php">
						<?php
						settings_fields( 'pwf_filter_fields' );
						do_settings_sections( 'pwf_filter_fields' );
						submit_button();
						?>
					</form>

					<form method="post" name="pwf-button-settings">
					<p class="pwf-buttons">
						<input type="submit" name="generate-pretty-urls" id="generate-pretty-urls" class="button button-large" value="<?php esc_html_e( 'Regenerate pretty URLs', 'pwf-woo-filter' ); ?>">
						<input type="submit" name="delete-cache-transients" id="delete-cache-transients" class="button button-large" value="<?php esc_html_e( 'Clear cached terms', 'pwf-woo-filter' ); ?>">
					</p>
					</form>
				</div>
			<?php
		}

		public function setup_sections() {
			add_settings_section(
				'pwf_filters_main_section',
				'',
				'',
				'pwf_filter_fields'
			);
		}

		public function setup_fields() {
			$fields = array();

			$fields[] = array(
				'name'    => esc_html__( 'Enable Pretty URLs', 'pwf-woo-filter' ),
				'id'      => 'pwf_shop_enable_pretty_links',
				'default' => 'enable',
				'type'    => 'select',
				'options' => self::get_enable_disable_options(),
			);
			$fields[] = array(
				'name'        => esc_html__( 'Pretty URLs prefixed', 'pwf-woo-filter' ),
				'desc'        => esc_html__( 'Fixing the conflict between the plugin pretty URLs links and the website sitemap or product taxonomies link.', 'pwf-woo-filter' ),
				'id'          => 'pwf_shop_pretty_urls_prefixed',
				'type'        => 'text',
				'placeholder' => 'filters',
			);
			$fields[] = array(
				'name'              => esc_html__( 'Transient time', 'pwf-woo-filter' ),
				'desc'              => esc_html__( 'Set transient time in seconds.', 'pwf-woo-filter' ),
				'id'                => 'pwf_transient_time',
				'autoload'          => false,
				'default'           => '86400',
				'type'              => 'number',
				'custom_attributes' => array(
					'min'  => 60,
					'step' => 60,
				),
			);

			$theme_desc    = '';
			$current_theme = get_template();
			if ( in_array( $current_theme, Pwf_Main::compitable_theme(), true ) ) {
				$theme_data = wp_get_theme();
				$theme_name = $theme_data->get( 'Template' );
				if ( empty( $theme_name ) ) {
					$theme_name = $theme_data->get( 'Name' );
				}
				$theme_desc = 'The plugin PWF adds custom codes to integrate with the <strong>' . ucfirst( $theme_name ) . '</strong> theme.';
			}
			$fields[] = array(
				'name'    => esc_html__( 'Theme Comitable', 'pwf-woo-filter' ),
				'id'      => 'pwf_shop_theme_compitable',
				'type'    => 'select',
				'options' => self::get_enable_disable_options(),
				'desc'    => $theme_desc,
			);
			$fields[] = array(
				'name'    => esc_html__( 'Analytics', 'pwf-woo-filter' ),
				'id'      => 'pwf_shop_analytics',
				'type'    => 'select',
				'options' => self::get_enable_disable_options(),
			);
			$fields[] = array(
				'name'    => esc_html__( 'User ID', 'pwf-woo-filter' ),
				'desc'    => esc_html__( 'Save The ID for a login user in the analtics database.', 'pwf-woo-filter' ),
				'id'      => 'pwf_shop_analytics_save_user_id',
				'type'    => 'select',
				'options' => self::get_enable_disable_options(),
			);

			if ( Pwf_Main::is_development_site() ) {
				$desc = esc_html__( 'This is a development website you don\'t need to add a purchase code here.', 'pwf-woo-filter' );
			} else {
				$desc = esc_html__( 'Please insert your Envato Purchase Code.', 'pwf-woo-filter' );
			}
			$fields[] = array(
				'name'        => esc_html__( 'Envato Purchase Code', 'pwf-woo-filter' ),
				'desc'        => $desc,
				'id'          => 'envato_purchase_code_28181010',
				'type'        => 'purchasecode',
				'desc_tip'    => esc_html__( 'Confirm that, according to the Envato License Terms, each license entitles one person for a single project. Creating multiple unregistered installations is a copyright violation.', 'pwf-woo-filter' ),
				'placeholder' => 'purchase code',
			);

			foreach ( $fields as $field ) {

				add_settings_field(
					$field['id'],
					$field['name'],
					array( $this, 'field_callback' ),
					'pwf_filter_fields',
					'pwf_filters_main_section',
					$field
				);

				register_setting( 'pwf_filter_fields', $field['id'] );
			}
		}

		public function field_callback( $arguments ) {
			$value = get_option( $arguments['id'] );

			if ( ! $value && isset( $arguments['default'] ) ) {
				$value = $arguments['default'];
			}

			switch ( $arguments['type'] ) {
				case 'text':
				case 'password':
				case 'number':
					$placeholder = $arguments['placeholder'] ?? '';
					printf(
						'<input name="%1$s" id="%1$s" type="%2$s" placeholder="%3$s" value="%4$s" />',
						esc_attr( $arguments['id'] ),
						esc_attr( $arguments['type'] ),
						esc_attr( $placeholder ),
						esc_attr( $value ),
					);
					break;
				case 'textarea':
					printf(
						'<textarea name="%1$s" id="%1$s" placeholder="%2$s" rows="5" cols="50">%3$s</textarea>',
						esc_attr( $arguments['id'] ),
						esc_attr( $arguments['placeholder'] ),
						$value
					);
					break;
				case 'select':
				case 'multiselect':
					if ( ! empty( $arguments['options'] ) && is_array( $arguments['options'] ) ) {
						$attributes     = '';
						$options_markup = '';
						if ( ! is_array( $value ) ) {
							$value = array( $value );
						}
						foreach ( $arguments['options'] as $key => $label ) {
								$options_markup .= sprintf(
									'<option value="%s" %s>%s</option>',
									esc_attr( $key ),
									selected( $value[ array_search( $key, $value, true ) ], $key, false ),
									esc_attr( $label )
								);
						}
						if ( 'multiselect' === $arguments['type'] ) {
							$attributes = ' multiple="multiple" ';
						}
						printf(
							'<select name="%1$s" id="%1$s" %2$s>%3$s</select>',
							esc_attr( $arguments['id'] ),
							$attributes,
							$options_markup
						);
					}
					break;
				case 'radio':
				case 'checkbox':
					if ( ! empty( $arguments['options'] ) && is_array( $arguments['options'] ) ) {
						$options_markup = '';
						$iterator       = 0;
						foreach ( $arguments['options'] as $key => $label ) {
							$iterator++;
							$options_markup .= sprintf(
								'<label for="%1$s_%6$s"><input id="%1$s_%6$s" name="%1$s[]" type="%2$s" value="%3$s" %4$s /> %5$s</label><br/>',
								esc_attr( $arguments['id'] ),
								esc_attr( $arguments['type'] ),
								esc_attr( $key ),
								checked( $value[ array_search( $key, $value, true ) ], $key, false ),
								esc_attr( $label ),
								esc_attr( $iterator )
							);
						}
						printf( '<fieldset>%s</fieldset>', $options_markup );
					}
					break;
				case 'purchasecode':
					$purchase_code = Pwf_Main::get_purchase_code();
					$value         = '';
					$css_verify    = '';
					$css_delete    = '';

					if ( ! empty( $purchase_code ) ) {
						$css_verify         = ' hidden';
						$value              = substr( $purchase_code, 0, 5 ) . 'xxxxx';
						$arguments['desc'] .= '<br/><strong>You can click the delete button to deactivate your license code from this domain if you are going to transfer your website to some other domain or server.</strong>';
					} else {
						$css_delete = ' hidden';
					}
					?>
					<input
						name="<?php echo esc_attr( $arguments['id'] ); ?>"
						id="<?php echo esc_attr( $arguments['id'] ); ?>"
						type="text"
						value="<?php echo esc_attr( $value ); ?>"
						class="pwf-purchasecode"
						placeholder="<?php echo esc_attr( $arguments['placeholder'] ); ?>"
						/>
						<button class="verify-purchase-code button<?php echo esc_attr( $css_verify ); ?>">Verfiy</button>
						<button class="delete-purchase-code button<?php echo esc_attr( $css_delete ); ?>">delete</button>
						<p class="purchase-code-message"></p>
					<?php
					break;
			}

			if ( isset( $arguments['desc'] ) && ! empty( $arguments['desc'] ) ) {
				printf(
					'<p class="description">%s</p>',
					wp_kses_post( $arguments['desc'] )
				);
			}
		}

		/**
		 * @param string $value
		 * @param string $option name
		 * @param string old value
		 */
		public function sanitize_fields( $value, $option, $old_value ) {

			switch ( $option ) {
				case 'pwf_shop_pretty_urls_prefixed':
					if ( ! empty( $value ) ) {
						$value = sanitize_key( $value );
						flush_rewrite_rules();
					}
					break;
				case 'envato_purchase_code_28181010':
					$value = $old_value;
					break;
				case 'pwf_transient_time':
					if ( ! empty( $value ) ) {
						$value = absint( $value );
					}
					break;
				case 'pwf_shop_enable_pretty_links':
				case 'pwf_shop_theme_compitable':
				case 'pwf_shop_analytics':
				case 'pwf_shop_analytics_save_user_id':
					if ( ! empty( $value ) ) {
						$value = sanitize_key( $value );
					}
					break;
			}

			return $value;
		}

		/**
		 * Define actions for two buttons
		 * 1- Regenerate pretty URLs
		 * 2- clear the plugin professional Woocommerce filter term counts transients cache.
		 *
		 * @since 1.67
		 */
		protected static function check_buttons() {
			//phpcs:ignore WordPress.Security.NonceVerification
			if ( isset( $_POST['generate-pretty-urls'] ) ) {
				Pwf_Routes_Manager::regenerates_routes();
				$message = esc_html__( 'Regenerate Pretty URLs is done.', 'pwf-woo-filter' );
				self::add_settings_error( $message, 'pwf-button-generate-pretty-urls', 'success' );
			}

			if ( isset( $_POST['delete-cache-transients'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification
				Pwf_Main::delete_transients();
				$message = esc_html__( 'Cached terms count cleared', 'pwf-woo-filter' );
				self::add_settings_error( $message, 'pwf-button-delete-cache-transients', 'success' );
			}
		}

		/**
		 * @since 1.7.0, 1.8.7
		 */
		public function admin_notice_license_activation() {
			if ( ! current_user_can( 'manage_options' ) ) {
				return;
			}

			if ( Pwf_Admin_Main::is_plugin_activated() ) {
				return;
			}

			if ( false !== get_transient( 'dismissed_plugin_pwf_woo_filter' ) ) {
				return;
			}

			$option_page = admin_url( 'edit.php?post_type=pwf_woofilter&page=pwf-settings' );

			echo '<div class="notice notice-warning is-dismissible">
			<p>Hi, Would you like to unlock premium features, Please activate your copy of 
			<strong><a href="' . esc_url( $option_page ) . '">PWF - WooCommerce Product Filters</strong></a>.
			<span style="display: block; margin: 0.5em 0.5em 0 0; clear: both;">
			<a href="?pwf-woocommerce-plugin-dismissed">Dismiss this notice</a>
			</span></p></div>';
		}

		public function notice_dismissed() {
			if ( ! current_user_can( 'manage_options' ) ) {
				return;
			}
			//phpcs:ignore WordPress.Security.NonceVerification.Recommended
			if ( isset( $_GET['pwf-woocommerce-plugin-dismissed'] ) ) {
				set_transient( 'dismissed_plugin_pwf_woo_filter', 1, 864000 );
			}
		}
	}

	Pwf_Admin_Settings_Page::register();
}
