<?php

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'Pwf_Main' ) ) {

	class Pwf_Main {

		/**
		* The unique instance of the plugin.
		*
		* @var Pwf_Main
		*/
		private static $instance;
		private static $is_network_plugin;
		private static $plugin_name = 'pwfwoofilter/pwfwoofilter.php';

		/**
		 * Gets an instance of our plugin.
		 *
		 * @return WP_Kickass_Plugin
		 */
		public static function get_instance() {
			if ( null === self::$instance ) {
				self::$instance = new self();
			}

			return self::$instance;
		}

		/**
		 * @since 1.0.0, 1.0.6
		 */
		private function __construct() {
			add_action( 'init', array( $this, 'init' ), 10 );
			add_action( 'init', array( $this, 'register_filter_post_type' ), 10 );
			add_action( 'init', array( $this, 'add_compitable_to__current_theme' ), 10 );
			add_action( 'init', array( $this, 'check_plugins_compatible' ), 1000 );
			add_action( 'rest_api_init', array( $this, 'prepare_filter_post_for_api' ), 10 );
			add_action( 'rest_prepare_pwf_woofilter', array( $this, 'remove_extra_data_from_filter_post_api' ), 12, 3 );

			// Update plugin cached
			add_action( 'edit_post', array( $this, 'update_terms_cached' ), 10, 2 );
			add_action( 'rest_delete_product', array( $this, 'update_terms_cached_api_delete' ), 10, 1 );

			add_filter( 'wp_kses_allowed_html', array( $this, 'wp_kses_allowed_html_tag' ), 10, 1 );

			add_action( 'customize_register', array( $this, 'add_customizer_section' ), 10, 1 );
			add_action( 'customize_preview_init', array( $this, 'add_customizer_preview_js' ), 10 );
			add_action( 'update_option_pwf_style', array( $this, 'delete_css_customizer_file' ), 10 );

			add_action( 'wp_trash_post', array( $this, 'wp_trash_post' ), 10, 1 );
			add_action( 'delete_term', array( $this, 'delete_term' ), 10, 4 );

			/**
			 * Set it before the class parse request run
			 */
			Pwf_Filter_Manager::set_pwf_global_variables();
		}

		public function init() {
			load_plugin_textdomain( 'pwf-woo-filter', false, PWF_WOO_FILTER_DIR_DOMAIN . '/languages/' );
			add_shortcode( 'pwf_filter', array( $this, 'add_shortcode' ) );
			add_shortcode( 'pwf_filter_seo_short_description', array( $this, 'add_shortcode_seo_short_description' ) );
			add_shortcode( 'pwf_filter_seo_h1_title', array( $this, 'add_shortcode_seo_h1_title' ) );
			add_shortcode( 'pwf_filter_active_filters', array( $this, 'add_shortcode_active_filters' ) );
			if ( class_exists( 'DGWT_WC_Ajax_Search' ) ) {
				new Pwf_Integrate_Ajax_Search_For_Woo();
			}

		}

		public function add_customizer_section( $wp_customize ) {
			$customizer = new Pwf_Customizer( $wp_customize );
		}

		public function add_customizer_preview_js() {
			$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
			wp_enqueue_script( 'pwf-preview', PWF_WOO_FILTER_URI . '/assets/js/admin/customizer-preview' . $suffix . '.js', array( 'jquery' ), PWF_WOO_FILTER_VER, true );
		}

		public function delete_css_customizer_file() {
			Pwf_Generate_Css::delete_css_file();
		}

		public static function get_purchase_code() {
			if ( self::check_is_network_plugin() ) {
				return get_site_option( 'envato_purchase_code_pwf_28181010', '' );
			} else {
				return get_option( 'envato_purchase_code_pwf_28181010', '' );
			}
		}

		/**
		 * @since 1.4.4
		 */
		public static function is_purchase_code() {
			$purchase_code = self::get_purchase_code();
			if ( ! empty( $purchase_code ) ) {
				return true;
			}

			return false;
		}

		/**
		 * @since 1.5.3
		 */
		public static function is_development_site() {
			$host = self::get_site_url();

			$chunks = explode( '.', $host );

			if ( 1 === count( $chunks ) ) {
				return true;
			}

			$dev_sites = array( 'dev', 'local', 'test', 'wp', 'example', 'invalid', 'localhost' );

			if ( in_array( end( $chunks ), $dev_sites, true ) ) {
				return true;
			}

			if ( preg_match( '/^[0-9\.]+$/', $host ) ) {
				return true;
			}

			return false;
		}

		/**
		 * @since 1.4.4
		 */
		public static function get_site_url() {
			if ( self::check_is_network_plugin() ) {
				return network_site_url();
			} else {
				return site_url();
			}
		}

		/**
		 * @since 1.4.4
		 */
		public static function check_is_network_plugin() {

			if ( is_null( self::$is_network_plugin ) ) {
				// Check is PWF as network plugin
				if ( is_multisite() ) {
					if ( ! function_exists( 'is_plugin_active_for_network' ) ) {
						require_once ABSPATH . '/wp-admin/includes/plugin.php';
					}
					if ( is_plugin_active_for_network( self::$plugin_name ) || is_network_only_plugin( self::$plugin_name ) ) {
						self::$is_network_plugin = true;
						return true;
					}
				}
			} else {
				return self::$is_network_plugin;
			}

			return false;
		}

		/**
		 * Used to check is_purchase_code && is_development_site
		 *
		 * @since 1.8.0
		 */
		public static function is_plugin_activated() {
			return ( self::is_development_site() || self::is_purchase_code() );
		}

		public function register_filter_post_type() {
			$labels = array(
				'name'               => esc_html_x( 'Filter', 'post type general name', 'pwf-woo-filter' ),
				'singular_name'      => esc_html_x( 'Filter', 'post type singular name', 'pwf-woo-filter' ),
				'menu_name'          => esc_html_x( 'Filters', 'admin menu', 'pwf-woo-filter' ),
				'name_admin_bar'     => esc_html_x( 'Filter', 'add new on admin bar', 'pwf-woo-filter' ),
				'add_new'            => esc_html_x( 'Add new', 'Filter', 'pwf-woo-filter' ),
				'add_new_item'       => esc_html__( 'Add new Filter', 'pwf-woo-filter' ),
				'new_item'           => esc_html__( 'New Filter', 'pwf-woo-filter' ),
				'edit_item'          => esc_html__( 'Edit Filter', 'pwf-woo-filter' ),
				'view_item'          => esc_html__( 'View Filter', 'pwf-woo-filter' ),
				'all_items'          => esc_html__( 'Filters', 'pwf-woo-filter' ),
				'search_items'       => esc_html__( 'Search Filters', 'pwf-woo-filter' ),
				'parent_item_colon'  => esc_html__( 'Parent Filters:', 'pwf-woo-filter' ),
				'not_found'          => esc_html__( 'No Filters found.', 'pwf-woo-filter' ),
				'not_found_in_trash' => esc_html__( 'No Filters found in trash.', 'pwf-woo-filter' ),
			);

			$post_type_args = array(
				'public'              => false,
				'exclude_from_search' => true,
				'publicly_queryable'  => false,
				'show_ui'             => true,
				'show_in_nav_menus'   => true,
				'hierarchical'        => false,
				'supports'            => false,
				'capability_type'     => 'post',
				'rewrite'             => false,
				'query_var'           => false,
				'has_archive'         => false,
				'label'               => 'Filter',
				'labels'              => $labels,
				'show_in_rest'        => true,
				'menu_icon'           => 'dashicons-filter',
			);

			register_post_type( 'pwf_woofilter', $post_type_args );
		}

		public function add_shortcode( $atts ) {
			// @codingStandardsIgnoreLine
			extract( shortcode_atts( array( 'id' => '' ), $atts ) );
			if ( ! absint( $id ) ) {
				return;
			}

			return wp_kses_post( Pwf_Filter_Manager::excute_plugin_shortcode_widget( $id ) );
		}

		/**
		 * Shortcode to display Seo short description
		 *
		 * @since 1.8.0
		 *
		 * @return string|empty
		 */
		public function add_shortcode_seo_short_description( $atts ) {
			ob_start();
			Pwf_Seo::seo_short_description();
			return ob_get_clean();
		}

		/**
		 * Shortcode to display Seo short description
		 *
		 * @since 1.8.3
		 *
		 * @return string|empty
		 */
		public function add_shortcode_seo_h1_title( $atts ) {
			ob_start();
			Pwf_Seo::modify_h1_title();
			return ob_get_clean();
		}

		/**
		 * Shortcode for active filters display on front-end
		 */
		public function add_shortcode_active_filters( $atts ) {
			self::output_active_filters();
		}

		/**
		 * Output active filters wrap
		 */
		public static function output_active_filters() {
			echo '<div class="pwf-woo-filter-active-filters"></div>';
		}

		public function prepare_filter_post_for_api() {

			register_rest_field(
				'pwf_woofilter',
				'title',
				array(
					'get_callback' => array( $this, 'get_title' ),
					'schema'       => null,
				)
			);

			register_rest_field(
				'pwf_woofilter',
				'filter_items',
				array(
					'get_callback' => array( $this, 'get_filter_items' ),
					'schema'       => null,
				)
			);
		}

		public function get_title( $object ) {
			return esc_attr( get_the_title( $object['id'] ) );
		}

		public function get_filter_items( $object ) {
			$render_filter = new Pwf_Render_Filter( $object['id'] );
			$filter_items  = new Pwf_Api_Prepare_Filter_Post( $render_filter->get_filter_items_data(), $render_filter->get_filter_setting() );
			$filter_items  = $filter_items->get_filter_items();

			return $filter_items;
		}

		public function remove_extra_data_from_filter_post_api( $data, $post, $context ) {
			if ( ! is_wp_error( $data ) ) {
				unset( $data->data['type'] );
				unset( $data->data['status'] );
				unset( $data->data['slug'] );
				unset( $data->data['date'] );
				unset( $data->data['date_gmt'] );
				unset( $data->data['modified'] );
				unset( $data->data['modified_gmt'] );
				unset( $data->data['link'] );
				unset( $data->data['template'] );
				unset( $data->data['guid'] );
			}
			return $data;
		}

		public function wp_kses_allowed_html_tag( $tags ) {
			$div = array(
				'aria-required' => true,
			);

			$input = array(
				'id'            => true,
				'type'          => true,
				'name'          => true,
				'class'         => true,
				'value'         => true,
				'placeholder'   => true,
				'aria-required' => true,
				'checked'       => true,
				'disabled'      => true,
				'min'           => true,
				'max'           => true,
				'data-*'        => true,
			);

			$select = array(
				'id'                 => true,
				'name'               => true,
				'class'              => true,
				'aria-required'      => true,
				'data-default-value' => true,
				'multiple'           => true,
			);

			$option = array(
				'value'      => true,
				'selected'   => true,
				'data-slug'  => true,
				'data-title' => true,
				'disabled'   => true,
			);

			if ( isset( $tags['div'] ) ) {
				$tags['div'] = array_merge( $tags['div'], $div );
			} else {
				$tags['div'] = $div;
			}

			if ( isset( $tags['input'] ) ) {
				$tags['input'] = array_merge( $tags['input'], $input );
			} else {
				$tags['input'] = $input;
			}

			if ( isset( $tags['select'] ) ) {
				$tags['select'] = array_merge( $tags['select'], $select );
			} else {
				$tags['select'] = $select;
			}

			if ( isset( $tags['option'] ) ) {
				$tags['option'] = array_merge( $tags['option'], $option );
			} else {
				$tags['option'] = $option;
			}

			$tags['optgroup'] = array(
				'value' => true,
				'label' => true,
			);

			return $tags;
		}

		/**
		 * Delete shop archive or pages realted to filter
		 *
		 * @since 1.5.4, 1.6.8
		 */
		public function wp_trash_post( $post_id ) {
			$post_ids = array();
			//phpcs:ignore WordPress.Security.NonceVerification.Recommended
			if ( isset( $_GET['post'] ) && is_array( $_GET['post'] ) ) {
				//phpcs:ignore WordPress.Security.NonceVerification.Recommended
				foreach ( $_GET['post'] as $post_id ) {
					array_push( $post_ids, absint( $post_id ) );
				}
			} else {
				$post_ids = array( absint( $post_id ) );
			}

			$post = get_post( $post_ids[0], 'ARRAY_A' );

			$is_option_change  = false;
			$filter_option_ids = get_option( 'pwf_woo_query_filters', '' );
			if ( ! empty( $filter_option_ids ) ) {
				if ( 'pwf_woofilter' === $post['post_type'] ) {
					$filter_value_ids = array_map( 'absint', array_values( $filter_option_ids ) );
					foreach ( $post_ids as $post_id ) {
						$post_id = absint( $post_id );
						if ( in_array( $post_id, $filter_value_ids, true ) ) {
							$is_option_change  = true;
							$filter_option_ids = Pwf_Meta::remove_filter_id_from_saved_option( $filter_option_ids, $post_id );
						}
					}
				} elseif ( 'page' === $post['post_type'] ) {
					$filter_key_ids = array_keys( $filter_option_ids );
					foreach ( $post_ids as $post_id ) {
						$post_id = 'page__' . $post_id;
						if ( in_array( $post_id, $filter_key_ids, true ) ) {
							$is_option_change = true;
							unset( $filter_option_ids[ $post_id ] );
						}
					}
				}

				if ( $is_option_change ) {
					update_option( 'pwf_woo_query_filters', $filter_option_ids, 'no' );
				}
			}
		}

		/**
		 * Delete term from option name pwf_woo_main_query_filters
		 *
		 * @since 1.5.4, 1.7.0
		 */
		public function delete_term( $term, $tt_id, $taxonomy, $deleted_term ) {
			$filter_option_ids = get_option( 'pwf_woo_query_filters', '' );
			if ( ! empty( $filter_option_ids ) ) {
				$tax_object = get_taxonomy( $taxonomy );
				$tax_object = $tax_object->object_type;
				if ( isset( $tax_object[0] ) ) {
					$post_type      = $tax_object[0];
					$term_name      = $post_type . '__' . $taxonomy . '__' . $term;
					$filter_key_ids = array_keys( $filter_option_ids );
					if ( in_array( $term_name, $filter_key_ids, true ) ) {
						unset( $filter_option_ids[ $term_name ] );
						update_option( 'pwf_woo_query_filters', $filter_option_ids, 'no' );
					}
				}
			}
		}

		public function update_terms_cached( $post_id, $post ) {
			if ( 'product' === $post->post_type ) {
				self::delete_terms_cached();
			}
		}

		public function update_terms_cached_api_delete( $post ) {
			if ( 'product' === $post->post_type ) {
				self::delete_terms_cached();
			}
		}

		private static function delete_terms_cached() {
			$delete = apply_filters( 'pwf_filter_delete_cache_after_update_product_post', true );
			if ( true === $delete ) {
				self::delete_transients();
			}
		}

		/**
		 * Delete cached terms.
		 *
		 * @param string $transient_name
		 *
		 * @since 1.6.7
		 *
		 * @return void
		 */
		public static function delete_transients( $transient_name = '_transient_timeout_pwf_woo_filter_' ) {
			global $wpdb;

			$str_search = '%' . $wpdb->esc_like( $transient_name ) . '%';
			$transients = $wpdb->get_col(
				$wpdb->prepare(
					"SELECT option_name FROM {$wpdb->options} WHERE option_name LIKE %s",
					$str_search
				)
			);

			foreach ( $transients as $transient ) {
				$key = str_replace( '_transient_timeout_', '', $transient );
				delete_transient( $key );
			}

			// But guess what?  Sometimes transients are not in the DB, so we have to do this too:
			wp_cache_flush();
		}

		/**
		 * WordPress Themes require to add specific code
		 *
		 * @since 1.0.0, 1.6.7
		 */
		public static function compitable_theme() {
			$themes = array(
				'Avada',
				'dt-the7',
				'astra',
				'brooklyn',
				'consulting',
				'ekommart',
				'enfold',
				'freshio',
				'greenmart',
				'kallyas',
				'legenda',
				'merchandiser',
				'oceanwp',
				'upstore',
				'medizin',
				'puca',
				'konte',
				'atelier',
				'thefox',
				'dfd-ronneby',
				'martfury',
				'urna',
				'mediacenter',
				'stockie',
				'ark',
				'theretailer',
				'ninezeroseven',
				'stockholm',
				'hongo',
				'movedo',
				'royal',
				'ciyashop',
				'x',
				'exponent',
				'pofo',
				'cartzilla',
				'pinkmart',
				'flatsome',
				'woodmart',
				'uncode',
				'mobimax',
			);

			return $themes;
		}
		public function add_compitable_to__current_theme() {
			$compitable = get_option( 'pwf_shop_theme_compitable', 'enable' );
			if ( 'disable' === $compitable ) {
				return;
			}
			$current_theme = get_template();
			if ( in_array( $current_theme, self::compitable_theme(), true ) ) {
				$path = PWF_WOO_FILTER_DIR . 'includes/compitable-themes/' . $current_theme . '/theme.php';
				Pwf_Autoloader::load_file( $path );
			}
		}

		/**
		 * You can check active plugins that compatible with pwf plugin
		 */
		public function check_plugins_compatible() {
			self::is_active_plugin_power_pack_for_elementor();
			self::is_active_plugin_elementor_pro();
			self::is_active_plugin_woocommerce_single_variations();
		}

		/**
		 * Add support to the plugin Power Pack pro for elementor
		 * @since ver 1.3.9
		 */
		protected static function is_active_plugin_power_pack_for_elementor() {
			if ( defined( 'POWERPACK_ELEMENTS_VER' ) ) {
				$powerpack_elements = new Pwf_Powerpack_Elements();
			}
		}

		/**
		 * Add support to the plugin Elementor Pro
		 * @since ver 1.3.9
		 */
		protected static function is_active_plugin_elementor_pro() {
			if ( defined( 'ELEMENTOR_PRO_VERSION' ) ) {
				$elementor_pro = new Pwf_Elementor_Pro();
			}
		}

		/**
		 * Check if the plugin active WooCommerce Single Variations
		 *
		 * @since 1.6.6
		 */
		protected static function is_active_plugin_woocommerce_single_variations() {
			if ( class_exists( 'WooCommerce_Single_Variations' ) ) {
				$integrate = apply_filters( 'pwf_woo_integrate_woocommerce_single_variations', false );
				if ( $integrate ) {
					new Pwf_Woocommerce_Single_Variations();
				}
			}
		}
	}

	$pwf_core = Pwf_Main::get_instance();
}
